! Spherical harmonic discrete ordinate radiative transfer method
! for plane-parallel atmospheres modified for data assimilation:
!   Subroutines for the forward computations.



SUBROUTINE CALCULATE_OPTICAL_PROPERTIES (NLAY, NCOMP, ML, SPECIESFLAG, &
                                         LWP, RMM, MOLECTAU, TAURAYL, &
                                         SRCTYPE, SOLARMU, NOUT, MUOUT,PHIOUT,&
                                         TAUP, ALBEDOP, LEGENP, SSPHASE, &
                               NSCATTAB, SPECIES2TABLE, MAXNR, NRTAB, RTAB, &
                               EXTINCTTAB, SSALBTAB, MAXLEG, NLEGTAB, LEGENTAB)
 ! Calculates the optical properties for the layers from the integrated 
 ! mass content (LWP) and mean mass radius (RMM) for the NCOMP particle 
 ! components at each layer.  The optical properties of the particle types
 ! (components) are added together including the molecular layer optical 
 ! depth.  The optical properties for each particle type are obtained by 
 ! linearly interpolating in radius in the input scattering tables.  
 ! If there is a solar source (SRCTYPE /= 'T') then the average 
 ! scattering phase function for each layer for the single scattering angle 
 ! defined by the solar direction and the outgoing radiance direction is 
 ! calculated (SSPHASE).
 !
 ! Inputs:
 !  NLAY        number of input property layers
 !  NCOMP       number of particle types
 !  ML          spherical harmonics L order
 !  SPECIESFLAG flags for each component and layer telling whether that 
 !                particle type exists for that layer (.TRUE.) or not (.FALSE.)
 !  LWP         integrated mass content for each component and layer
 !                (g/m^2, if scattering tables are normalized to 1 g/m^3)
 !  RMM        mean mass radius for each component and layer 
 !                (micron, if scattering tables are in microns)
 !  MOLECTAU    molecular absorption optical depth for each layer
 !  TAURAYL     Rayleigh molecular scattering optical depth for each layer
 !  SRCTYPE     'S' for solar source, 'T' for thermal source, 'B' for both
 !                (used to determine whether to compute SSPHASE)
 !  SOLARMU     negative of cosine of solar zenith angle (< 0)
 !                direct beam is assumed to be going in phi=0 azimuth
 !  NOUT        number of output radiance directions  (for computing SSPHASE)
 !  MUOUT       cosine zenith angles for radiance output
 !  PHIOUT      azimuth angles (degrees) for radiance output
 !
 ! Scattering table inputs:
 !  NSCATTAB    number of scattering tables
 !  SPECIES2TABLE array of indices to convert from component to scattering table
 !  MAXNR       maximum number of radii over all tables
 !  NRTAB       number of radii in each table
 !  RTAB        radii list (um) for each table
 !  EXTINCTTAB  extinctions for each radius in each table (/km for LWC=1 g/m^3)
 !  SSALBTAB    single scattering albedos for each radius in each table
 !  MAXLEG      maximum order of Legendre series over all tables
 !  NLEGTAB     Legendre order in each table
 !  LEGENTAB    Legendre series coefficients (l=1...NLEGTAB) for each radius
 !                in each table.  l=0 coef is 1, l=1 coeff is 3*g.
 !
 ! Outputs:
 !  TAUP      optical depths of the input layers (from top down)
 !  ALBEDOP   single scattering albedo of the inputs layers
 !  LEGENP    phase function Legendre coefficients (l=1...ML+1) for each layer
 !  SSPHASE   phase functions for NOUT solar-viewing geometries for each layer

  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NCOMP, NLAY, ML, NOUT, NSCATTAB, MAXNR, MAXLEG
  INTEGER, INTENT(IN) :: SPECIES2TABLE(NCOMP)
  LOGICAL, INTENT(IN) :: SPECIESFLAG(NCOMP,NLAY)
  REAL,    INTENT(IN) :: SOLARMU, MUOUT(NOUT), PHIOUT(NOUT)
  REAL,    INTENT(IN) :: MOLECTAU(NLAY), TAURAYL(NLAY)
  REAL,    INTENT(IN) :: LWP(NCOMP,NLAY), RMM(NCOMP,NLAY)
  INTEGER, INTENT(IN) :: NRTAB(NSCATTAB), NLEGTAB(MAXNR,NSCATTAB)
  REAL,    INTENT(IN) :: RTAB(MAXNR,NSCATTAB)
  REAL,    INTENT(IN) :: EXTINCTTAB(MAXNR,NSCATTAB), SSALBTAB(MAXNR,NSCATTAB)
  REAL,    INTENT(IN) :: LEGENTAB(1:MAXLEG,MAXNR,NSCATTAB)
  CHARACTER(LEN=1), INTENT(IN) :: SRCTYPE
  REAL,    INTENT(OUT) :: TAUP(NLAY), ALBEDOP(NLAY), LEGENP(1:ML+1,NLAY)
  REAL,    INTENT(OUT) :: SSPHASE(NLAY,NOUT)
  INTEGER :: I, IT, J, K, KL, KM, KU, L, NLEG
  REAL    :: F, EXT, SSALB, TAU, TAU_TOT, TAU_SCA, P1, P2
  REAL    :: TAUEPS=0.0E-9, TAUTINY=1.0E-12
  DOUBLE PRECISION :: COSSCAT, RADDEG, PHIRAD
  DOUBLE PRECISION, ALLOCATABLE :: SUNDIRLEG(:,:)


  ALLOCATE (SUNDIRLEG(0:MAXLEG,NOUT))
  IF (SRCTYPE /= 'T') THEN
    ! Compute the Legendre polynomials for the scattering angle 
    !   for the untruncated solar single scattering computation.
    RADDEG = ACOS(-1.0D0)/180.0D0
    DO J = 1, NOUT
      PHIRAD = PHIOUT(J)*RADDEG
      COSSCAT = SOLARMU*MUOUT(J) &
              + SQRT((1.0D0-SOLARMU**2)*(1.0D0-MUOUT(J)**2))*COS(PHIRAD)
      CALL LEGENDRE_ALL (COSSCAT, MAXLEG, SUNDIRLEG(:,J))
      SUNDIRLEG(:,J) = SUNDIRLEG(:,J)/(4.0D0*ACOS(-1.0D0))
    ENDDO
  ENDIF

  DO L = 1, NLAY
     ! Initialize optical properties with molecular absorption and scattering
    TAU_TOT = MOLECTAU(L) + TAURAYL(L)
    TAU_SCA = TAURAYL(L)
    LEGENP(1:ML+1,L) = 0.0
    LEGENP(2,L) = LEGENP(2,L) + TAURAYL(L)*0.5
    SSPHASE(L,:) = 0.0
    IF (SRCTYPE /= 'T') THEN
      DO J = 1, NOUT
        SSPHASE(L,J)=SSPHASE(L,J)+TAURAYL(L)*(SUNDIRLEG(0,J)+0.5*SUNDIRLEG(2,J))
      ENDDO
    ENDIF
    NLEG = 0
     ! Loop over particle types in each layer
    DO I = 1, NCOMP
      IT = SPECIES2TABLE(I)
      IF (SPECIESFLAG(I,L) .AND. IT > 0) THEN
         ! Do binary search to find location of radius in table
        KL = 1   
        KU = NRTAB(IT)
        DO WHILE (KU-KL > 1)
          KM = (KU+KL)/2
          IF (RMM(I,L) >= RTAB(KM,IT)) THEN
            KL = KM
          ELSE
            KU = KM
          ENDIF
        ENDDO
        K = KL
         ! Interpolate scattering properties
        F = (RMM(I,L)-RTAB(K,IT))/(RTAB(K+1,IT)-RTAB(K,IT))
        F = MIN(1.0,MAX(0.0,F))
        EXT = (1-F)*EXTINCTTAB(K,IT) + F*EXTINCTTAB(K+1,IT)
        TAU = LWP(I,L)*EXT*0.001 + TAUEPS
        SSALB = (1-F)*SSALBTAB(K,IT) + F*SSALBTAB(K+1,IT)
         ! Add in the properties for this component
        TAU_TOT = TAU_TOT + TAU
        TAU_SCA = TAU_SCA + TAU*SSALB
        LEGENP(1:ML+1,L) = LEGENP(1:ML+1,L) + TAU*SSALB  &
               *( (1-F)*LEGENTAB(1:ML+1,K,IT) + F*LEGENTAB(1:ML+1,K+1,IT) )
        IF (SRCTYPE /= 'T') THEN
         ! If there is a solar source then compute the phase function
         ! at the single scattering angle defined by the solar beam direction
         ! and each outgoing radiance direction.
          DO J = 1, NOUT
            P1 = SUNDIRLEG(0,J) + SUM( SUNDIRLEG(1:NLEGTAB(K,IT),J) *LEGENTAB(1:NLEGTAB(K,IT),K,IT) )
            P2 = SUNDIRLEG(0,J) + SUM( SUNDIRLEG(1:NLEGTAB(K+1,IT),J) *LEGENTAB(1:NLEGTAB(K+1,IT),K+1,IT) )
            SSPHASE(L,J) = SSPHASE(L,J) + TAU*SSALB*( (1-F)*P1 + F*P2 )
          ENDDO
        ENDIF
      ENDIF
    ENDDO
     ! Normalize to get the single scattering albedo and Legendre coefs
    SSALB = TAU_SCA/(TAU_TOT+TAUTINY)
    TAUP(L) = TAU_TOT
    ALBEDOP(L) = SSALB
    LEGENP(1:ML+1,L) = LEGENP(1:ML+1,L)/(TAU_SCA+TAUTINY)
    SSPHASE(L,:) = SSPHASE(L,:)/(TAU_SCA+TAUTINY)
    IF (ANY(SSPHASE(L,:) < 0.0)) THEN
      WRITE (*,*) 'COMPUTE_OPTICAL_PROPERTIES: negative phase function for layer: ',L
      STOP
    ENDIF
  ENDDO

  DEALLOCATE (SUNDIRLEG)
END SUBROUTINE CALCULATE_OPTICAL_PROPERTIES





SUBROUTINE SOLVE_ADAPTIVE_SHDOMPP (NLAY, TEMPP, TAUP, ALBEDOP, LEGENP, &
                               MAXIG, ML, MM, NLM, NMU, NPHI, NPHI0MAX, &
                               SRCTYPE, SOLARFLUX, SOLARMU, &
                               WAVELEN, WAVENO, SKYRAD, &
                               SFCTEMP, SFCTYPE, SFCPARMS, &
                               MAXDELTAU, SPLITACC, MAXITER, ACCELFLAG, PRNT, &
                               NPTS, FRACTAUG, IXP, SHSOURCE, &
                               SOLCRIT, ITER)
 ! Performs the adaptive grid portion of the SHDOMPP solution procedure to 
 ! calculate plane-parallel monochromatic, unpolarized radiative transfer.
 ! The inputs are the optical properties, uniform in each layer, the 
 ! surface properties, and the control parameters.  The output are the 
 ! spherical harmonic series for the source function at the adaptive grid 
 ! levels, and the adaptive grid with the fractional optical depths within 
 ! each input layer.
 !
 ! Inputs:
 !  NLAY      number of input property layers (TAUP, ALBEDOP, etc.)
 !  TEMPP     temperatures (K) of input layer interfaces (NLAY+1 values)
 !              temperature is linearly interpolated in tau within each layer
 !  TAUP      optical depths of the input layers (each layer from top down)
 !  ALBEDOP   single scattering albedo of the inputs layers
 !  LEGENP    phase function Legendre coefficients (l=1...ML+1) for each layer
 !  MAXIG     maximum possible size of adaptive grid arrays
 !  ML        spherical harmonics L order
 !  MM        spherical harmonics M order
 !  NLM       number of spherical harmonics terms
 !  NMU       number of discrete ordinate zenith angles in both hemispheres
 !  NPHI      number of discrete ordinate azimuth angles in 2\pi
 !  NPHI0MAX  number of discrete ordinate azimuth angles actually used
 !  SRCTYPE   'S' for solar source, 'T' for thermal source, 'B' for both
 !  SOLARFLUX solar flux on a *horizontal* surface
 !  SOLARMU   negative of cosine of solar zenith angle (< 0)
 !              direct beam is assumed to be going in phi=0 azimuth
 !  WAVELEN   wavelength (microns) for thermal emission
 !  WAVENO(2) starting and ending wavenumber (cm^-1) for thermal (if WAVELEN<=0)
 !  SKYRAD    top of atmosphere downward sky radiance (or temp for SRCTYPE='T')
 !  SFCTEMP   surface temperature (K)
 !  SFCTYPE   surface type (currently 'L' for Lambertian, 'R' for RPV
 !  SFCPARMS  array of surface parameters (first is albedo for Lambertian)
 !  MAXDELTAU max sublayer scattering optical depth in adaptive grid
 !  SPLITACC  adaptive layer splitting accuracy
 !  MAXITER   maximum number of iterations allowed
 !  ACCELFLAG logical flag: true to perform iteration acceleration
 !  PRNT      logical flag: true to print iteration information
 !
 ! Outputs: 
 !  NPTS      number of adaptive grid points
 !  FRACTAUG  sorted grid point optical depth array for the adaptive grid
 !  IXP       index of starting and ending in FRACTAUG for each input layer
 !  SHSOURCE  spherical harmonic series of source function at each level
 !  SOLCRIT   actual solution accuracy achieved
 !  ITER      actual number of iterations
 !
 !  FRACTAUG(IXP(1,layer)) = 0   FRACTAUG(IXP(2,layer)) = 1
 !  There are two output grid levels in each input layer (at the boundaries)
 ! for layers with scaled optical depth less than MINDELTAUSPLIT, and at
 ! least two levels for those input layers with greater optical depth.
 ! Thus each input boundary has two grid levels, which is needed because
 ! the source function may be discontinuous across the layers (since the
 ! optical properties are discontinuous).

  IMPLICIT NONE
  INTEGER, INTENT(IN)  :: NLAY, MAXIG
  INTEGER, INTENT(IN)  :: ML, MM, NLM, NMU, NPHI, NPHI0MAX, MAXITER
  LOGICAL, INTENT(IN)  :: ACCELFLAG, PRNT
  REAL,    INTENT(IN)  :: TAUP(NLAY), ALBEDOP(NLAY), LEGENP(1:ML+1,NLAY)
  REAL,    INTENT(IN)  :: TEMPP(NLAY+1)
  REAL,    INTENT(IN)  :: SOLARFLUX, SOLARMU, WAVELEN, WAVENO(2), SKYRAD
  REAL,    INTENT(IN)  :: SFCPARMS(*), SFCTEMP
  REAL,    INTENT(IN)  :: MAXDELTAU, SPLITACC
  CHARACTER(LEN=1), INTENT(IN) :: SRCTYPE, SFCTYPE
  INTEGER, INTENT(OUT) :: NPTS, IXP(2,NLAY), ITER
  REAL,    INTENT(OUT) :: FRACTAUG(MAXIG), SHSOURCE(NLM,MAXIG)
  REAL,    INTENT(OUT) :: SOLCRIT

  INTEGER :: ORDINATESET, OLDNPTS, PREVNPTS, LASTNPTS, NANG, I, L
  LOGICAL, PARAMETER :: DELTAM=.TRUE.
  REAL,    PARAMETER :: MINDELTAUSPLIT=1.0E-3
  REAL    :: DELJDOT, DELJOLD, DELJNEW
  REAL    :: SOLARAZ, PI, GNDALB, DIRFLUX
  REAL    :: TRUESOLCRIT, CURSPLITACC, ACCPAR
  INTEGER, ALLOCATABLE :: IXG(:), NPHI0(:)
  REAL, ALLOCATABLE :: TAUPSC(:), ALBEDOSC(:), LEGENSC(:,:)
  REAL, ALLOCATABLE :: TAUGSC(:), RADIANCE(:,:,:), SOURCE(:,:)
  REAL, ALLOCATABLE :: MU(:), WTDO(:,:), PHI(:,:), WTMU(:), YLMSUN(:)
  REAL, ALLOCATABLE :: CMU1(:,:), CMU2(:,:), CPHI1(:,:,:), CPHI2(:,:,:)
  REAL, ALLOCATABLE :: SRCINT(:,:,:)
  REAL, ALLOCATABLE :: CONSTSOURCE(:,:), SHRADIANCE(:,:), DOSOURCE(:,:,:)
  REAL, ALLOCATABLE :: DELSOURCE(:,:)


  ALLOCATE (TAUPSC(NLAY), ALBEDOSC(NLAY), LEGENSC(1:ML,NLAY))
  ALLOCATE (TAUGSC(MAXIG), IXG(MAXIG))
  ALLOCATE (MU(NMU), PHI(NPHI0MAX,NMU), NPHI0(NMU), WTDO(NMU,NPHI0MAX))
  ALLOCATE (YLMSUN(NLM), WTMU(NMU), CMU1(NLM,NMU), CMU2(NMU,NLM))
  ALLOCATE (CPHI1(0:MM,NPHI0MAX,NMU), CPHI2(NPHI0MAX,0:MM,NMU))
  ALLOCATE (SOURCE(NLM,MAXIG), RADIANCE(NPHI0MAX,NMU,MAXIG))
  ALLOCATE (CONSTSOURCE(NLM,MAXIG), DELSOURCE(NLM,MAXIG))
  ALLOCATE (SRCINT(0:4,NMU,MAXIG))


  ! Set up some things before solution loop
 
  ! If Delta-M then scale the optical depth, albedo, and Legendre terms.
  IF (DELTAM) THEN
    CALL DELTA_SCALE (NLAY, ML, TAUP, ALBEDOP, LEGENP, &
                      TAUPSC, ALBEDOSC, LEGENSC)
  ELSE
    TAUPSC(:) = TAUP(:)
    ALBEDOSC(:) = ALBEDOP(:)
    LEGENSC(1:ML,:) = LEGENP(1:ML,:)
  ENDIF

  CALL MAKE_TAU_GRID (NLAY, TAUPSC, ALBEDOSC, MAXDELTAU, MINDELTAUSPLIT,&
                      NPTS, TAUGSC, IXP, IXG)

   ! Abort if no adaptive grid
  IF (SPLITACC <= 0.0 .OR. MAXITER <= 0) GOTO 900

  ! Precompute Ylm's for solar direction
  IF (SRCTYPE /= 'T') THEN
    SOLARAZ = 0.0
    CALL YLMALL (SOLARMU, SOLARAZ, ML, MM, 1, YLMSUN)
  ENDIF
  ! Prepare the constant part of the level source function, which are
  !   the Planck blackbody source and the solar attenuated single scattered
  !   radiance, as a function of level.
  CONSTSOURCE(1,1:NPTS) = -1.0
  CALL PREPARE_SOURCE (NLAY, TAUPSC, ALBEDOSC, TEMPP, LEGENSC, &
                       SRCTYPE, ML, MM, NLM, NPTS, TAUGSC, IXP, IXG, &
                       SOLARFLUX, SOLARMU, YLMSUN, WAVELEN, WAVENO, &
                       CONSTSOURCE)

  ! Make the discrete ordinates (angles) 
  !   (set 2 is reduced gaussian, 3 is reduced double gauss)
  ORDINATESET = 3
  CALL MAKE_ANGLE_SET (NMU, NPHI, 1, NPHI0MAX, ORDINATESET, &
                       NPHI0, MU, PHI, WTMU, WTDO, NANG)

  ! Make the Ylm transform coefficients
  CALL MAKE_SH_DO_COEF (ML, MM, NLM, NMU, NPHI0, &
                        NPHI0MAX, MU, PHI, WTMU, WTDO, &
                        CMU1, CMU2, CPHI1, CPHI2)

  ! Initialize the radiance on the base grid using Eddington 
  !   two-stream plane-parallel
  IF (SFCTYPE == 'L') THEN
    GNDALB = SFCPARMS(1)
  ELSE
    GNDALB = 0.2
  ENDIF
  ALLOCATE (SHRADIANCE(NLM,NPTS))
  CALL INIT_RADIANCE (NLAY, ML, TAUPSC, ALBEDOSC, LEGENSC, TEMPP, &
                      SRCTYPE, SOLARFLUX, SOLARMU, GNDALB, SFCTEMP, & 
                      SKYRAD, WAVELEN,  WAVENO, NPTS, TAUGSC, IXP, NLM, &
                      SHRADIANCE)

  ! Initialize the SH source function from the SH radiance field
  OLDNPTS = 0
  SOURCE(:,:) = 0.0
  CALL COMPUTE_SOURCE (ML,MM, NLM, NLAY, NPTS, IXP, IXG, &
                       ALBEDOSC, LEGENSC, &
                       SHRADIANCE, CONSTSOURCE, SOURCE, &
                       OLDNPTS, DELSOURCE, &
                       DELJDOT, DELJOLD, DELJNEW, SOLCRIT)
  DEALLOCATE (SHRADIANCE)

  SOLCRIT = 1.0
  TRUESOLCRIT = 1.0
  CURSPLITACC = 1.0
  LASTNPTS = -1
  OLDNPTS = NPTS
  ACCPAR = 0.0
  RADIANCE(:,:,:) = 0.0
  DELSOURCE(:,:) = 0.0
  ITER = 0

  IF (PRNT) WRITE (*,*) '! Iter Log(Sol)  Log(True)  Npoints'
 
  ! Main solution loop
    ! Iterate until the iterations exceed the limit, or the desired
    ! solution criterion is reached 
  DO WHILE (ITER < MAXITER .AND. (CURSPLITACC > SPLITACC .OR. NPTS /= OLDNPTS))
    ITER = ITER + 1

    ! Transform the source from spherical harmonics to discrete ordinates
    ALLOCATE (DOSOURCE(NPHI0MAX,NMU,MAXIG))
    CALL SH_TO_DO (NPTS, ML, MM, NLM, NMU, NPHI0MAX, NPHI0, &
                   CMU1, CPHI1, SOURCE, DOSOURCE)
      ! Enforce the discrete ordinate source function to be non-negative
    DOSOURCE(:,:,:) = MAX(0.0,DOSOURCE(:,:,:))


    ! Keep on doing integrations as long as new levels are added
    OLDNPTS = NPTS
    PREVNPTS = 0
    DO WHILE (NPTS > PREVNPTS)
      ! Integrate the source function along discrete ordinates to
      ! compute the radiance field.  
      CALL PATH_INTEGRATION (NLAY, NPTS, IXP, IXG, TAUGSC, &
                             NMU, NPHI0MAX, NPHI0, &
                             SRCTYPE, SOLARFLUX, SOLARMU, SKYRAD, &
                             SFCTYPE, SFCPARMS, SFCTEMP, WAVELEN, WAVENO, &
                             MU, PHI, WTDO, SRCINT, LASTNPTS, &
                             DOSOURCE, RADIANCE)
      PREVNPTS = NPTS
      ! Do the adaptive grid cell splitting stuff if desired
      CALL SPLIT_CELLS (NLAY, NPTS, MAXIG, IXP, IXG, TAUGSC, &
                        NMU, NPHI0MAX, NPHI0, MU, DOSOURCE, RADIANCE, &
                        SPLITACC, TRUESOLCRIT, CURSPLITACC)
      IF (NPTS > PREVNPTS) THEN
        CONSTSOURCE(1,PREVNPTS+1:NPTS) = -1.0
        CALL PREPARE_SOURCE (NLAY, TAUPSC, ALBEDOSC, TEMPP, LEGENSC, &
                             SRCTYPE, ML, MM, NLM, NPTS, TAUGSC, IXP, IXG, &
                             SOLARFLUX, SOLARMU, YLMSUN, WAVELEN, WAVENO, &
                             CONSTSOURCE)
      ENDIF
    ENDDO
    DEALLOCATE (DOSOURCE)

    ! Transform the radiance from discrete ordinates to spherical harmonics
    ALLOCATE (SHRADIANCE(NLM,NPTS))
    CALL DO_TO_SH (NPTS, ML, MM, NLM, NMU, NPHI0MAX, NPHI0, &
                   CMU2, CPHI2, RADIANCE, SHRADIANCE)

    ! Calculate the source function from the radiance in spherical harmonics.
    !   Also computes the solution criterion and the series acceleration
    !   stuff (DELJDOT, DELJOLD, DELJNEW, DELSOURCE).
    CALL COMPUTE_SOURCE (ML,MM, NLM, NLAY, NPTS, IXP, IXG, &
                         ALBEDOSC, LEGENSC, &
                         SHRADIANCE, CONSTSOURCE, SOURCE, &
                         OLDNPTS, DELSOURCE, &
                         DELJDOT, DELJOLD, DELJNEW, SOLCRIT)
    DEALLOCATE (SHRADIANCE)

    ! Accelerate the convergence of the source function vector
    IF (ACCELFLAG) THEN
      CALL ACCELERATE_SOLUTION (NLM, NPTS, OLDNPTS, SOURCE, DELSOURCE, &
                                DELJDOT, DELJOLD, DELJNEW, ACCPAR)
    ENDIF
    IF (DELJNEW < DELJOLD) &
        TRUESOLCRIT = SOLCRIT/(1.0-SQRT(DELJNEW/DELJOLD))

    ! Print out the iteration, log solution criterion, and number of points
    IF (PRNT) WRITE (*,'(2X,I4,2F8.3,1X,I6)')  ITER, &
        LOG10(MAX(SOLCRIT,1.0E-20)), LOG10(MAX(TRUESOLCRIT,1.0E-20)), NPTS
  ENDDO

  IF (PRNT) WRITE (*,'(1X,A,I6,A,F9.6)') &
              '! Iterations: ', ITER, '     Final Criterion: ', SOLCRIT

900 CONTINUE

   ! Save the spherical harmonics source function in optical depth order
  DO I = 1, NPTS
    SHSOURCE(:,I) = SOURCE(:,IXG(I))
  ENDDO

  ! Make the array with the fractional optical depth in each input layer
  !   for each adaptive grid level
  DO L = 1, NLAY
    FRACTAUG(IXP(1,L)) = 0.0
    FRACTAUG(IXP(2,L)) = 1.0
    DO I = IXP(1,L)+1, IXP(2,L)-1
      IF (TAUGSC(IXP(2,L))-TAUGSC(IXP(1,L)) > 0.0) THEN
        FRACTAUG(I) = (TAUGSC(I) - TAUGSC(IXP(1,L))) &
                    / (TAUGSC(IXP(2,L)) - TAUGSC(IXP(1,L)))
      ELSE
        FRACTAUG(I) = (I - IXP(1,L)) / FLOAT(IXP(2,L)-IXP(1,L))
      ENDIF
    ENDDO
  ENDDO

  DEALLOCATE (TAUPSC, ALBEDOSC, LEGENSC, TAUGSC)
  DEALLOCATE (MU, PHI, NPHI0, WTDO, YLMSUN, WTMU)
  DEALLOCATE (CPHI1, CPHI2, CMU1, CMU2)
  DEALLOCATE (SOURCE, RADIANCE, CONSTSOURCE, DELSOURCE, SRCINT)
END SUBROUTINE SOLVE_ADAPTIVE_SHDOMPP




SUBROUTINE SOLVE_FIXED_SHDOMPP1 (NLAY, TEMPP, &
                                TAUP, ALBEDOP, LEGENP, SSPHASE, &
                                NPTS, FRACTAUG, IXP, SHSOURCE, &
                                ML, MM, NLM, NMU, NPHI, NPHI0MAX, &
                                SRCTYPE, SOLARFLUX, SOLARMU, &
                                WAVELEN, WAVENO, SKYRAD, &
                                SFCTEMP, SFCTYPE, SFCPARMS, &
                                SOLACC, MAXITER, ACCELFLAG, PRNT, &
                                SOLCRIT, ITER, &
                                NOUT, MUOUT, PHIOUT, &
                                RADOUT, NMOREITER, ACCELPAR)
 ! Performs the fixed grid portion of the SHDOMPP solution procedure to 
 ! calculate plane-parallel monochromatic, unpolarized radiative transfer.
 ! The inputs are the fixed grid level locations (FRACTAUG), the spherical
 ! harmonics source function series at each level, the input optical 
 ! properties, the surface properties, and the control parameters.  The 
 ! outputs are the vector of radiances in the specified directions.
 ! Use SOLVE_ADAPTIVE_SHDOMPP to perform the adaptive grid iterations
 ! and compute NPTS, FRACTAU, IXP, SHSOURCE, SOLCRIT, ITER which are
 ! inputs to this routine.
 !
 ! Inputs:
 !  NLAY      number of input property layers (TAUP, ALBEDOP, etc.)
 !  TEMPP     temperatures (K) of input layer interfaces (NLAY+1 values)
 !              temperature is linearly interpolated in tau within each layer
 !  TAUP      optical depths of the input layers (each layer from top down)
 !  ALBEDOP   single scattering albedo of the inputs layers
 !  LEGENP    phase function Legendre coefficients (l=1...ML+1) for each layer
 !  SSPHASE   phase functions for NOUT solar-viewing geometries for each layer
 !  NPTS      number of adaptive grid points
 !  FRACTAUG  fraction optical depth grid point array for the adaptive grid
 !  IXP       index of starting and ending in FRACTAUG for each input layer
 !  SHSOURCE  spherical harmonic series of source function at each level
 !  ML        spherical harmonics L order
 !  MM        spherical harmonics M order
 !  NLM       number of spherical harmonics terms
 !  NMU       number of discrete ordinate zenith angles in both hemispheres
 !  NPHI      number of discrete ordinate azimuth angles in 2\pi
 !  NPHI0MAX  number of discrete ordinate azimuth angles actually used
 !  SRCTYPE   'S' for solar source, 'T' for thermal source, 'B' for both
 !  SOLARFLUX solar flux on a *horizontal* surface
 !  SOLARMU   negative of cosine of solar zenith angle (< 0)
 !              direct beam is assumed to be going in phi=0 azimuth
 !  WAVELEN   wavelength (microns) for thermal emission
 !  WAVENO(2) starting and ending wavenumber (cm^-1) for thermal (if WAVELEN<=0)
 !  SKYRAD    top of atmosphere downward sky radiance (or temp for SRCTYPE='T')
 !  SFCTEMP   surface temperature (K)
 !  SFCTYPE   surface type (currently 'L' for Lambertian, 'R' for RPV
 !  SFCPARMS  array of surface parameters (first is albedo for Lambertian)
 !  SOLACC    desired solution accuracy (normalized iteration change in SOURCE)
 !  MAXITER   maximum number of iterations allowed
 !  ACCELFLAG logical flag: true to perform iteration acceleration
 !  PRNT      logical flag: true to print iteration information
 !  NOUT      number of output radiance directions
 !  MUOUT     cosine zenith angles for radiance output:
 !              If MUOUT>0 then the radiance is computed at the top of the
 !              atmosphere, and if MUOUT<0 it is computed for the surface.
 !  PHIOUT    azimuth angles (degrees) for radiance output
 !
 ! Input and output:
 !  SOLCRIT   In: solution accuracy output from SOLVE_ADAPTIVE_SHDOMPP
 !            Out: solution accuracy achieved after call
 !  ITER      In: number of iterations from SOLVE_ADAPTIVE_SHDOMPP
 !            Out: total number of iterations
 !
 ! Outputs: 
 !  RADOUT    vector of NOUT radiances
 !  NMOREITER number of iterations performed
 !  ACCELPAR  vector of acceleration step sizes
 ! 
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY
  INTEGER, INTENT(IN) :: NPTS, IXP(2,NLAY)
  INTEGER, INTENT(IN) :: ML, MM, NLM, NMU, NPHI, NPHI0MAX
  INTEGER, INTENT(IN) :: MAXITER, NOUT
  INTEGER, INTENT(OUT) :: NMOREITER
  LOGICAL, INTENT(IN) :: ACCELFLAG, PRNT
  CHARACTER(LEN=1), INTENT(IN) :: SRCTYPE, SFCTYPE
  REAL,    INTENT(IN) :: TEMPP(NLAY+1), TAUP(NLAY), ALBEDOP(NLAY)
  REAL,    INTENT(IN) :: LEGENP(1:ML+1,NLAY), SSPHASE(NLAY,NOUT)
  REAL,    INTENT(IN) :: FRACTAUG(NPTS), SHSOURCE(NLM,NPTS)
  REAL,    INTENT(IN) :: SOLARFLUX, SOLARMU, SKYRAD
  REAL,    INTENT(IN) :: SFCPARMS(*), SFCTEMP, WAVELEN, WAVENO(2)
  REAL,    INTENT(IN) :: MUOUT(NOUT), PHIOUT(NOUT)
  REAL,    INTENT(IN) :: SOLACC
  INTEGER, INTENT(INOUT) :: ITER
  REAL,    INTENT(INOUT) :: SOLCRIT
  REAL,    INTENT(OUT) :: RADOUT(NOUT), ACCELPAR(MAXITER)
  INTEGER :: I, J, K, L, M, ME, LAY, IT
  INTEGER :: NANG, ORDINATESET, LOFJ(NLM)
  LOGICAL, PARAMETER :: DELTAM=.TRUE.
  REAL    :: SOLARAZ, RADDEG, PHIRAD, ACCPAR
  REAL    :: DIRFLUXSFC, SFCPLANCK, BNDRAD, TEMP, PLANCKFUNC
  REAL    :: TAU, GNDALB, TRUESOLCRIT, DELJDOT, DELJOLD, DELJNEW
  INTEGER, ALLOCATABLE :: NPHI0(:)
  REAL, ALLOCATABLE :: MU(:), WTDO(:,:), PHI(:,:), WTMU(:), YLMSUN(:)
  REAL, ALLOCATABLE :: CMU1(:,:), CMU2(:,:), CPHI1(:,:,:), CPHI2(:,:,:)
  REAL, ALLOCATABLE :: SRCINT(:,:,:)
  REAL, ALLOCATABLE :: TAUPSC(:), ALBEDOSC(:), LEGENSC(:,:), DELTA_F(:)
  REAL, ALLOCATABLE :: DIRFLUX(:), DTAUGSC(:), SOURCE1(:), RADVEC(:)
  REAL, ALLOCATABLE :: CONSTSOURCE(:,:), DELSOURCE(:,:)
  REAL, ALLOCATABLE :: SOURCE(:,:), DOSOURCE(:,:,:)
  REAL, ALLOCATABLE :: RADIANCE(:,:,:), SHRADIANCE(:,:)


  ALLOCATE (MU(NMU), PHI(NPHI0MAX,NMU), NPHI0(NMU))
  ALLOCATE (WTMU(NMU), WTDO(NMU,NPHI0MAX))
  ALLOCATE (YLMSUN(NLM), CMU1(NLM,NMU), CMU2(NMU,NLM))
  ALLOCATE (CPHI1(0:MM,NPHI0MAX,NMU), CPHI2(NPHI0MAX,0:MM,NMU))
  ALLOCATE (TAUPSC(NLAY), ALBEDOSC(NLAY), LEGENSC(1:ML,NLAY), DELTA_F(NLAY))
  ALLOCATE (DTAUGSC(NPTS), DIRFLUX(NPTS))
  ALLOCATE (CONSTSOURCE(NLM,NPTS), DELSOURCE(NLM,NPTS))
  ALLOCATE (SRCINT(0:4,NMU,NPTS))
  ALLOCATE (SOURCE1(NPTS), RADVEC(NPTS))
  ALLOCATE (SOURCE(NLM,NPTS), DOSOURCE(NPHI0MAX,NMU,NPTS))
  ALLOCATE (SHRADIANCE(NLM,NPTS), RADIANCE(NPHI0MAX,NMU,NPTS))


  RADDEG = ACOS(-1.0)/180.0

  ! If Delta-M then scale the optical depth, albedo, and Legendre terms.
  IF (DELTAM) THEN
    CALL DELTA_SCALE (NLAY, ML, TAUP, ALBEDOP, LEGENP, &
                      TAUPSC, ALBEDOSC, LEGENSC)
    DELTA_F(:) = LEGENP(ML+1,:)/(2*(ML+1)+1)
  ELSE
    TAUPSC(:) = TAUP(:)
    ALBEDOSC(:) = ALBEDOP(:)
    LEGENSC(1:ML,:) = LEGENP(1:ML,:)
    DELTA_F(:) = 0.0
  ENDIF

   ! Use the fractional optical depth to make the sublayer delta tau's
  DO LAY = 1, NLAY
    DO I = IXP(1,LAY), IXP(2,LAY)-1
      DTAUGSC(I) = (FRACTAUG(I+1)-FRACTAUG(I))*TAUPSC(LAY)
    ENDDO
    DTAUGSC(IXP(2,LAY)) = 0.0
  ENDDO

   ! Make the l index as a function of SH term (J)
  J = 0
  DO L = 0, ML
    ME = MIN(L,MM)
    DO M = 0, ME
      J = J + 1
      LOFJ(J) = L
    ENDDO
  ENDDO

  ! Make the discrete ordinates (angles) 
  !   (set 2 is reduced gaussian, 3 is reduced double gauss)
  ORDINATESET = 3
  CALL MAKE_ANGLE_SET (NMU, NPHI, 1, NPHI0MAX, ORDINATESET, &
                       NPHI0, MU, PHI, WTMU, WTDO, NANG)

  ! Make the Ylm transform coefficients
  CALL MAKE_SH_DO_COEF (ML, MM, NLM, NMU, NPHI0, &
                        NPHI0MAX, MU, PHI, WTMU, WTDO, &
                        CMU1, CMU2, CPHI1, CPHI2)

  IF (SRCTYPE /= 'T') THEN
     ! Precompute Ylm's for solar direction
    SOLARAZ = 0.0
    CALL YLMALL (SOLARMU, SOLARAZ, ML, MM, 1, YLMSUN)

      ! Make the perpendicular direct beam flux for all the levels
    TAU = 0.0
    DO I = 1, NPTS
      DIRFLUX(I) = SOLARFLUX/ABS(SOLARMU) *EXP(-TAU/ABS(SOLARMU))
      TAU = TAU + DTAUGSC(I)
    ENDDO
    DIRFLUXSFC = DIRFLUX(NPTS)*ABS(SOLARMU)
  ENDIF

  ! Make the constant part of the level source function, which are
  !   the Planck blackbody source and the solar attenuated single scattered
  !   radiance, as a function of level.
  CALL MAKE_CONST_SOURCE (NLAY, ALBEDOSC, TEMPP, LEGENSC, &
                          SRCTYPE, ML, MM, NLM, NPTS, FRACTAUG, IXP, LOFJ, &
                          DIRFLUX, YLMSUN, WAVELEN, WAVENO, &
                          CONSTSOURCE)

   ! Compute the source integration coefficients from the optical depth array
  CALL CALC_SRCINT (NLAY, NPTS, IXP, DTAUGSC, NMU, MU, SRCINT)

   ! Start the source function off at the previously computed values
  SOURCE(:,:) = SHSOURCE(:,:)

  TRUESOLCRIT = 1.0
  RADIANCE(:,:,:) = 0.0
  DELSOURCE(:,:) = 0.0

  IF (PRNT) WRITE (*,*) '! Iter Log(Sol)  Log(True)  Npoints'

    ! Main solution loop
    ! Iterate until the iterations exceed the limit, or the desired
    ! solution criterion is reached 
  IT = 0
  ACCPAR = 0.0
  DO WHILE (IT == 0 .OR. (ITER < MAXITER .AND. SOLCRIT > SOLACC))
    ITER = ITER + 1
    IT = IT + 1

     ! Transform the source from spherical harmonics to discrete ordinates
    CALL SH_TO_DO (NPTS, ML, MM, NLM, NMU, NPHI0MAX, NPHI0, &
                   CMU1, CPHI1, SOURCE, DOSOURCE)

     ! Integrate the source function along discrete ordinates to
     ! compute the radiance field.  
    CALL PATH_INTEG (NLAY, NPTS, IXP,  &
                     NMU, NPHI0MAX, NPHI0, &
                     SRCTYPE, DIRFLUXSFC, SOLARMU, SKYRAD, &
                     SFCTYPE, SFCPARMS, SFCTEMP, WAVELEN, WAVENO, &
                     MU, PHI, WTDO, SRCINT, &
                     DOSOURCE, RADIANCE)

     ! Transform the radiance from discrete ordinates to spherical harmonics
    CALL DO_TO_SH (NPTS, ML, MM, NLM, NMU, NPHI0MAX, NPHI0, &
                   CMU2, CPHI2, RADIANCE, SHRADIANCE)

     ! Calculate the SH source function from the SH radiances
    CALL CALC_SOURCE (ML,MM, NLM, NLAY, NPTS, IXP, LOFJ, &
                      ALBEDOSC, LEGENSC, &
                      SHRADIANCE, CONSTSOURCE, SOURCE, &
                      DELSOURCE, DELJDOT, DELJOLD, DELJNEW, SOLCRIT)

     ! Accelerate the convergence of the source function vector
    IF (ACCELFLAG) THEN
      CALL ACCELERATE_SOLUTION (NLM, NPTS, NPTS, SOURCE, DELSOURCE, &
                                DELJDOT, DELJOLD, DELJNEW, ACCPAR)
      ACCELPAR(IT) = ACCPAR
    ELSE
      ACCELPAR(IT) = 0.0
    ENDIF
    IF (DELJNEW < DELJOLD) &
        TRUESOLCRIT = SOLCRIT/(1.0-SQRT(DELJNEW/DELJOLD))

     ! Print out the iteration, log solution criterion, and number of points
    IF (PRNT) WRITE (*,'(2X,I4,2F8.3,1X,I6,1X,F7.5)')  ITER, &
        LOG10(MAX(SOLCRIT,1.0E-20)), LOG10(MAX(TRUESOLCRIT,1.0E-20)), NPTS, ACCPAR
  ENDDO
  NMOREITER = IT

  IF (PRNT) WRITE (*,'(1X,A,I6,A,F9.6)') &
              '! Iterations: ', ITER, '     Final Criterion: ', SOLCRIT

! --------------------

  IF (SFCTYPE /= 'L') THEN
    IF (SRCTYPE /= 'T') THEN
      SFCPLANCK = 0.0
    ELSE
      SFCPLANCK = PLANCKFUNC(WAVELEN,WAVENO,SFCTEMP)
    ENDIF
  ENDIF


   ! Loop over the output radiance directions
  DO K = 1, NOUT
    PHIRAD = PHIOUT(K)*RADDEG

     ! Compute the source function throughout grid for this angle
    CALL COMPUTE_ONE_SOURCE (NLAY, ALBEDOSC, LEGENSC, SSPHASE(:,K), DELTA_F, &
                             NPTS, IXP, ML, MM, NLM, &
                             MUOUT(K), PHIRAD, SRCTYPE, DELTAM, &
                             SOLARMU, DIRFLUX, SOURCE, SOURCE1)

     ! Get boundary radiances: either top or bottom
     !  Isotropic top boundary or Lambertian bottom boundary can use
     !  the previously computed boundary radiances in RADIANCE,
     !  otherwise, compute the radiance for this angle by integrating 
     !  over the stored downwelling radiances.
    IF (MUOUT(K) < 0.0) THEN
      BNDRAD = RADIANCE(1,1,1)
    ELSE
      IF (SFCTYPE == 'L') THEN
        BNDRAD = RADIANCE(1,NMU/2+1,NPTS)
      ELSE
        CALL VARIABLE_BRDF_SURFACE (NMU, NPHI0MAX, NPHI0, MU, PHI, WTDO, &
                                    MUOUT(K), PHIRAD, &
                                    SRCTYPE, SOLARMU, DIRFLUXSFC, &
                                    SFCTYPE, SFCPARMS, SFCPLANCK, &
                                    RADIANCE(:,:,NPTS), BNDRAD)
      ENDIF
    ENDIF

     ! Integrate the source function for this direction to get the radiances
     !   at all the levels (RADVEC), then pull out the desired one.
    CALL INTEGRATE_SOURCE (NLAY, NPTS, IXP, DTAUGSC, &
                           MUOUT(K), BNDRAD, SOURCE1, RADVEC)
    IF (MUOUT(K) > 0) THEN
      RADOUT(K) = RADVEC(IXP(1,1))
    ELSE
      RADOUT(K) = RADVEC(IXP(2,NLAY))
    ENDIF
  ENDDO

  DEALLOCATE (MU, PHI, NPHI0, WTMU, WTDO, YLMSUN)
  DEALLOCATE (CMU1, CMU2, CPHI1, CPHI2)
  DEALLOCATE (TAUPSC, ALBEDOSC, LEGENSC, DELTA_F, DTAUGSC, DIRFLUX)
  DEALLOCATE (CONSTSOURCE, DELSOURCE, SOURCE, DOSOURCE, SHRADIANCE, RADIANCE)
  DEALLOCATE (SRCINT, SOURCE1, RADVEC)
END SUBROUTINE SOLVE_FIXED_SHDOMPP1






SUBROUTINE SOLVE_FIXED_SHDOMPP2 (NLAY, TEMPP, &
                                TAUP, ALBEDOP, LEGENP, SSPHASE, &
                                NPTS, FRACTAUG, IXP, SHSOURCE, &
                                ML, MM, NLM, NMU, NPHI, NPHI0MAX, &
                                SRCTYPE, SOLARFLUX, SOLARMU, &
                                WAVELEN, WAVENO, SKYRAD, &
                                SFCTEMP, SFCTYPE, SFCPARMS, &
                                NMOREITER, ACCELPAR, &
                                NOUT, MUOUT, PHIOUT, RADOUT)
 ! Performs the fixed grid portion of the SHDOMPP solution procedure to 
 ! calculate plane-parallel monochromatic, unpolarized radiative transfer.
 ! The inputs are the fixed grid level locations (FRACTAUG), the spherical
 ! harmonics source function series at each level, the input optical 
 ! properties, the surface properties, and the control parameters.  The 
 ! outputs are the vector of radiances in the specified directions.
 ! Use SOLVE_ADAPTIVE_SHDOMPP to perform the adaptive grid iterations
 ! and compute NPTS, FRACTAU, IXP, SHSOURCE, SOLCRIT, ITER which are
 ! inputs to this routine.
 !
 ! Inputs:
 !  NLAY      number of input property layers (TAUP, ALBEDOP, etc.)
 !  TEMPP     temperatures (K) of input layer interfaces (NLAY+1 values)
 !              temperature is linearly interpolated in tau within each layer
 !  TAUP      optical depths of the input layers (each layer from top down)
 !  ALBEDOP   single scattering albedo of the inputs layers
 !  LEGENP    phase function Legendre coefficients (l=1...ML+1) for each layer
 !  SSPHASE   phase functions for NOUT solar-viewing geometries for each layer
 !  NPTS      number of adaptive grid points
 !  FRACTAUG  fraction optical depth grid point array for the adaptive grid
 !  IXP       index of starting and ending in FRACTAUG for each input layer
 !  SHSOURCE  spherical harmonic series of source function at each level
 !  ML        spherical harmonics L order
 !  MM        spherical harmonics M order
 !  NLM       number of spherical harmonics terms
 !  NMU       number of discrete ordinate zenith angles in both hemispheres
 !  NPHI      number of discrete ordinate azimuth angles in 2\pi
 !  NPHI0MAX  number of discrete ordinate azimuth angles actually used
 !  SRCTYPE   'S' for solar source, 'T' for thermal source, 'B' for both
 !  SOLARFLUX solar flux on a *horizontal* surface
 !  SOLARMU   negative of cosine of solar zenith angle (< 0)
 !              direct beam is assumed to be going in phi=0 azimuth
 !  WAVELEN   wavelength (microns) for thermal emission
 !  WAVENO(2) starting and ending wavenumber (cm^-1) for thermal (if WAVELEN<=0)
 !  SKYRAD    top of atmosphere downward sky radiance (or temp for SRCTYPE='T')
 !  SFCTEMP   surface temperature (K)
 !  SFCTYPE   surface type (currently 'L' for Lambertian, 'R' for RPV
 !  SFCPARMS  array of surface parameters (first is albedo for Lambertian)
 !  NMOREITER number of iterations to perform
 !  ACCELPAR  acceleration step size to take for each iteration
 !  NOUT      number of output radiance directions
 !  MUOUT     cosine zenith angles for radiance output:
 !              If MUOUT>0 then the radiance is computed at the top of the
 !              atmosphere, and if MUOUT<0 it is computed for the surface.
 !  PHIOUT    azimuth angles (degrees) for radiance output
 !
 ! Outputs: 
 !  RADOUT    vector of NOUT radiances

  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY
  INTEGER, INTENT(IN) :: NPTS, IXP(2,NLAY)
  INTEGER, INTENT(IN) :: ML, MM, NLM, NMU, NPHI, NPHI0MAX
  INTEGER, INTENT(IN) :: NMOREITER, NOUT
  CHARACTER(LEN=1), INTENT(IN) :: SRCTYPE, SFCTYPE
  REAL,    INTENT(IN) :: TEMPP(NLAY+1), TAUP(NLAY), ALBEDOP(NLAY)
  REAL,    INTENT(IN) :: LEGENP(1:ML+1,NLAY), SSPHASE(NLAY,NOUT)
  REAL,    INTENT(IN) :: FRACTAUG(NPTS), SHSOURCE(NLM,NPTS)
  REAL,    INTENT(IN) :: SOLARFLUX, SOLARMU, SKYRAD
  REAL,    INTENT(IN) :: SFCPARMS(*), SFCTEMP, WAVELEN, WAVENO(2)
  REAL,    INTENT(IN) :: MUOUT(NOUT), PHIOUT(NOUT)
  REAL,    INTENT(IN) :: ACCELPAR(NMOREITER)
  REAL,    INTENT(OUT) :: RADOUT(NOUT)
  INTEGER :: I, I1, I2, J, K, L, M, ME, LAY, IT
  INTEGER :: NANG, ORDINATESET, LOFJ(NLM)
  LOGICAL, PARAMETER :: DELTAM=.TRUE.
  REAL    :: SOLARAZ, RADDEG, PHIRAD
  REAL    :: DIRFLUXSFC, SFCPLANCK, TEMP, PLANCKFUNC
  REAL    :: TAU, GNDALB
  INTEGER, ALLOCATABLE :: NPHI0(:)
  REAL, ALLOCATABLE :: MU(:), WTDO(:,:), PHI(:,:), WTMU(:), YLMSUN(:)
  REAL, ALLOCATABLE :: CMU1(:,:), CMU2(:,:), CPHI1(:,:,:), CPHI2(:,:,:)
  REAL, ALLOCATABLE :: SRCINT(:,:,:)
  REAL, ALLOCATABLE :: TAUPSC(:), ALBEDOSC(:), LEGENSC(:,:), DELTA_F(:)
  REAL, ALLOCATABLE :: DIRFLUX(:), DTAUGSC(:), TAUGSC(:)
  REAL, ALLOCATABLE :: SOURCE1(:,:), BNDRAD(:), RADVEC(:,:)
  REAL, ALLOCATABLE :: CONSTSOURCE(:,:)
  REAL, ALLOCATABLE :: SOURCE(:,:), ASOURCE(:,:), DOSOURCE(:,:,:)
  REAL, ALLOCATABLE :: RADIANCE(:,:,:), SHRADIANCE(:,:)

  IF (NMOREITER < 1) THEN
    PRINT *, 'SOLVE_FIXED_SHDOMPP2: NMOREITER must be at least 1'
    STOP
  ENDIF
  ALLOCATE (MU(NMU), PHI(NPHI0MAX,NMU), NPHI0(NMU))
  ALLOCATE (WTMU(NMU), WTDO(NMU,NPHI0MAX))
  ALLOCATE (YLMSUN(NLM), CMU1(NLM,NMU), CMU2(NMU,NLM))
  ALLOCATE (CPHI1(0:MM,NPHI0MAX,NMU), CPHI2(NPHI0MAX,0:MM,NMU))
  ALLOCATE (TAUPSC(NLAY), ALBEDOSC(NLAY), LEGENSC(1:ML,NLAY), DELTA_F(NLAY))
  ALLOCATE (DTAUGSC(NPTS), TAUGSC(NPTS), DIRFLUX(NPTS))
  ALLOCATE (CONSTSOURCE(NLM,NPTS))
  ALLOCATE (SRCINT(0:4,NMU,NPTS))
  ALLOCATE (SOURCE1(NPTS,NOUT), BNDRAD(NOUT), RADVEC(NPTS,NOUT))
  ALLOCATE (SOURCE(NLM,NPTS), ASOURCE(NLM,NPTS))
  ALLOCATE (DOSOURCE(NPHI0MAX,NMU,NPTS))
  ALLOCATE (SHRADIANCE(NLM,NPTS), RADIANCE(NPHI0MAX,NMU,NPTS))


  RADDEG = ACOS(-1.0)/180.0

  ! If Delta-M then scale the optical depth, albedo, and Legendre terms.
  IF (DELTAM) THEN
    CALL DELTA_SCALE (NLAY, ML, TAUP, ALBEDOP, LEGENP, &
                      TAUPSC, ALBEDOSC, LEGENSC)
    DELTA_F(:) = LEGENP(ML+1,:)/(2*(ML+1)+1)
  ELSE
    TAUPSC(:) = TAUP(:)
    ALBEDOSC(:) = ALBEDOP(:)
    LEGENSC(1:ML,:) = LEGENP(1:ML,:)
    DELTA_F(:) = 0.0
  ENDIF

   ! Use the fractional optical depth to make the sublayer delta tau's
  DO LAY = 1, NLAY
    DO I = IXP(1,LAY), IXP(2,LAY)-1
      DTAUGSC(I) = (FRACTAUG(I+1)-FRACTAUG(I))*TAUPSC(LAY)
    ENDDO
    DTAUGSC(IXP(2,LAY)) = 0.0
  ENDDO

   ! Make the l index as a function of SH term (J)
  J = 0
  DO L = 0, ML
    ME = MIN(L,MM)
    DO M = 0, ME  
      J = J + 1   
      LOFJ(J) = L
    ENDDO
  ENDDO

  ! Make the discrete ordinates (angles) 
  !   (set 2 is reduced gaussian, 3 is reduced double gauss)
  ORDINATESET = 3
  CALL MAKE_ANGLE_SET (NMU, NPHI, 1, NPHI0MAX, ORDINATESET, &
                       NPHI0, MU, PHI, WTMU, WTDO, NANG)

  ! Make the Ylm transform coefficients
  CALL MAKE_SH_DO_COEF (ML, MM, NLM, NMU, NPHI0, &
                        NPHI0MAX, MU, PHI, WTMU, WTDO, &
                        CMU1, CMU2, CPHI1, CPHI2)

  IF (SRCTYPE /= 'T') THEN
     ! Precompute Ylm's for solar direction
    SOLARAZ = 0.0
    CALL YLMALL (SOLARMU, SOLARAZ, ML, MM, 1, YLMSUN)

      ! Make the perpendicular direct beam flux for all the levels
    TAUGSC(1) = 0.0
    DIRFLUX(1) = SOLARFLUX/ABS(SOLARMU)
    DO I = 2, NPTS
      TAUGSC(I) = TAUGSC(I-1) + DTAUGSC(I-1)
      DIRFLUX(I) = SOLARFLUX/ABS(SOLARMU) *EXP(-TAUGSC(I)/ABS(SOLARMU))
    ENDDO
    DIRFLUXSFC = DIRFLUX(NPTS)*ABS(SOLARMU)
  ENDIF

  ! Make the constant part of the level source function, which are
  !   the Planck blackbody source and the solar attenuated single scattered
  !   radiance, as a function of level.
  CALL MAKE_CONST_SOURCE (NLAY, ALBEDOSC, TEMPP, LEGENSC, &
                          SRCTYPE, ML, MM, NLM, NPTS, FRACTAUG, IXP, LOFJ, &
                          DIRFLUX, YLMSUN, WAVELEN, WAVENO, &
                          CONSTSOURCE)

   ! Compute the source integration coefficients from the optical depth array
  CALL CALC_SRCINT (NLAY, NPTS, IXP, DTAUGSC, NMU, MU, SRCINT)

   ! Start the source function off at the previously computed values
  ASOURCE(:,:) = SHSOURCE(:,:)

!$TAF INIT tape_solve_fixed_shdompp2 = MEMORY

    ! Main solution loop
  DO IT = 1, NMOREITER

     ! Transform the source from spherical harmonics to discrete ordinates
    CALL SH_TO_DO (NPTS, ML, MM, NLM, NMU, NPHI0MAX, NPHI0, &
                   CMU1, CPHI1, ASOURCE, DOSOURCE)

     ! Integrate the source function along discrete ordinates to
     ! compute the radiance field.  
!$TAF STORE DOSOURCE = tape_solve_fixed_shdompp2
    CALL PATH_INTEG (NLAY, NPTS, IXP, &
                     NMU, NPHI0MAX, NPHI0, &
                     SRCTYPE, DIRFLUXSFC, SOLARMU, SKYRAD, &
                     SFCTYPE, SFCPARMS, SFCTEMP, WAVELEN, WAVENO, &
                     MU, PHI, WTDO, SRCINT,  DOSOURCE, RADIANCE)

     ! Transform the radiance from discrete ordinates to spherical harmonics
    CALL DO_TO_SH (NPTS, ML, MM, NLM, NMU, NPHI0MAX, NPHI0, &
                   CMU2, CPHI2, RADIANCE, SHRADIANCE)

     ! Calculate the SH source function from the SH radiances
!$TAF STORE SHRADIANCE = tape_solve_fixed_shdompp2
    CALL CALC_SOURCE2 (ML,MM, NLM, NLAY, NPTS, IXP, LOFJ, &
                       ALBEDOSC, LEGENSC, &
                       SHRADIANCE, CONSTSOURCE,  SOURCE)

     ! Accelerate the convergence of the source function vector
    ASOURCE(:,:) = SOURCE(:,:) + ACCELPAR(IT)*(SOURCE(:,:)-ASOURCE(:,:))
  ENDDO

! --------------------

  IF (SFCTYPE /= 'L') THEN
    IF (SRCTYPE /= 'T') THEN
      SFCPLANCK = 0.0
    ELSE
      SFCPLANCK = PLANCKFUNC(WAVELEN,WAVENO,SFCTEMP)
    ENDIF
  ENDIF


   ! Loop over the output radiance directions
  DO K = 1, NOUT
    PHIRAD = PHIOUT(K)*RADDEG

     ! Compute the source function throughout grid for this angle
    CALL COMPUTE_ONE_SOURCE (NLAY, ALBEDOSC, LEGENSC, SSPHASE(:,K), DELTA_F, &
                             NPTS, IXP, ML, MM, NLM, &
                             MUOUT(K), PHIRAD, SRCTYPE, DELTAM, &
                             SOLARMU, DIRFLUX, ASOURCE, SOURCE1(:,K))

     ! Get boundary radiances: either top or bottom
     !  Isotropic top boundary or Lambertian bottom boundary can use
     !  the previously computed boundary radiances in RADIANCE,
     !  otherwise, compute the radiance for this angle by integrating 
     !  over the stored downwelling radiances.
    IF (MUOUT(K) < 0.0) THEN
      BNDRAD(K) = RADIANCE(1,1,1)
    ELSE
      IF (SFCTYPE == 'L') THEN
        BNDRAD(K) = RADIANCE(1,NMU/2+1,NPTS)
      ELSE
        CALL VARIABLE_BRDF_SURFACE (NMU, NPHI0MAX, NPHI0, MU, PHI, WTDO, &
                                    MUOUT(K), PHIRAD, &
                                    SRCTYPE, SOLARMU, DIRFLUXSFC, &
                                    SFCTYPE, SFCPARMS, SFCPLANCK, &
                                    RADIANCE(:,:,NPTS), BNDRAD(K))
      ENDIF
    ENDIF

     ! Integrate the source function for this direction to get the radiances
     !   at all the levels (RADVEC), then pull out the desired one.
    CALL INTEGRATE_SOURCE (NLAY, NPTS, IXP, DTAUGSC, &
                           MUOUT(K), BNDRAD(K), SOURCE1(:,K), RADVEC(:,K))
    IF (MUOUT(K) > 0) THEN
      RADOUT(K) = RADVEC(IXP(1,1),K)
    ELSE
      RADOUT(K) = RADVEC(IXP(2,NLAY),K)
    ENDIF
  ENDDO

  DEALLOCATE (MU, PHI, NPHI0, WTMU, WTDO, YLMSUN)
  DEALLOCATE (CMU1, CMU2, CPHI1, CPHI2)
  DEALLOCATE (TAUPSC, ALBEDOSC, LEGENSC, DELTA_F, DTAUGSC, TAUGSC, DIRFLUX)
  DEALLOCATE (CONSTSOURCE, SOURCE, ASOURCE, DOSOURCE, SHRADIANCE, RADIANCE)
  DEALLOCATE (SRCINT, SOURCE1, BNDRAD, RADVEC)
END SUBROUTINE SOLVE_FIXED_SHDOMPP2




SUBROUTINE MAKE_CONST_SOURCE (NLAY, ALBEDOSC, TEMPP, LEGENSC, &
                              SRCTYPE, ML, MM, NLM, NPTS, FRACTAUG, IXP, LOFJ, &
                              DIRFLUX, YLMSUN, WAVELEN, WAVENO, &
                              CONSTSOURCE)
 ! Prepares the constant part of the source function for each level.
 ! This is either or both thermal emission or the single scattering
 ! solar pseudo-source.  Only those levels (points) with CONSTSOURCE(1,:)
 ! less than zero are changed, which allows this to be run for each time
 ! adaptive grid points are added.
 ! The thermal emission source is \sqrt{4\pi}(1-\omega)B(T), where the
 ! temperature T is assumed to vary linearly in optical depth.
 ! The solar source is 
 !   S_0/\mu_0 e^{-\tau/\mu_0} Y_{lm}(\mu_0,\phi_0) {\omega \chi_l \over 2l+1}
 ! where S_0 is the solar flux on a horizontal plane, \chi_l are the Legendre
 ! coefficients, and \tau is the optical depth at the level.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY, ML, MM, NLM, NPTS
  INTEGER, INTENT(IN) :: IXP(2,NLAY), LOFJ(NLM)
  REAL,    INTENT(IN) :: YLMSUN(NLM), WAVELEN, WAVENO(2)
  REAL,    INTENT(IN) :: ALBEDOSC(NLAY), LEGENSC(1:ML,NLAY)
  REAL,    INTENT(IN) :: TEMPP(NLAY+1), FRACTAUG(NPTS), DIRFLUX(NPTS)
  REAL,    INTENT(INOUT) :: CONSTSOURCE(NLM,NPTS)
  CHARACTER(LEN=1), INTENT(IN) :: SRCTYPE
  INTEGER :: I, J, K, L, LAY, ME, M
  REAL    :: C, TEMP, PLANCKFUNC

  C = SQRT(4.0*ACOS(-1.0))
  DO LAY = 1, NLAY
    DO K = IXP(1,LAY), IXP(2,LAY)
       CONSTSOURCE(:,K) = 0.0
       IF (SRCTYPE /= 'S') THEN
          TEMP = TEMPP(LAY) + FRACTAUG(K)*(TEMPP(LAY+1)-TEMPP(LAY))
          CONSTSOURCE(1,K) = C*(1.0-ALBEDOSC(LAY)) &
                            *PLANCKFUNC(WAVELEN,WAVENO,TEMP)
       ENDIF
       IF (SRCTYPE /= 'T') THEN
          CONSTSOURCE(1,K) = CONSTSOURCE(1,K)+DIRFLUX(K)*YLMSUN(1)*ALBEDOSC(LAY)
          DO J = 2, NLM
            L = LOFJ(J)
            CONSTSOURCE(J,K) = CONSTSOURCE(J,K) &
                 + DIRFLUX(K)*YLMSUN(J)*ALBEDOSC(LAY)*LEGENSC(L,LAY)/(2*L+1)
          ENDDO
       ENDIF
    ENDDO
  ENDDO
END SUBROUTINE MAKE_CONST_SOURCE



SUBROUTINE CALC_SOURCE (ML,MM, NLM, NLAY, NPTS, IXP, LOFJ, &
                        ALBEDOSC, LEGENSC, &
                        SHRADIANCE, CONSTSOURCE, SOURCE, &
                        DELSOURCE, DELJDOT, DELJOLD, DELJNEW, SOLCRIT)
 ! Computes the source function (SOURCE) in spherical harmonic space 
 ! for all the grid levels in all the input layers.  
 ! The thermal source and/or solar pseudo-sources in CONSTSOURCE is
 ! added to the scattering source (computed from LEGENP and the spherical 
 ! harmonic expansion in SHRADIANCE).
 ! This routine also computes some things for the sequence
 ! acceleration: the dot products of the difference in source function 
 ! between successive iterations and the new difference source function 
 ! vector (DELSOURCE).  
 ! Computes the new solution criterion (SOLCRIT, the RMS difference
 ! in succesive source function fields normalized by the RMS of the field
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: ML, MM, NLM, NLAY, NPTS
  INTEGER, INTENT(IN) :: IXP(2,NLAY), LOFJ(NLM)
  REAL,    INTENT(IN) :: ALBEDOSC(NLAY), LEGENSC(1:ML,NLAY)
  REAL,    INTENT(IN) :: SHRADIANCE(NLM,NPTS), CONSTSOURCE(NLM,NPTS)
  REAL,    INTENT(INOUT) :: SOURCE(NLM,NPTS), DELSOURCE(NLM,NPTS)
  REAL,    INTENT(OUT) :: DELJDOT, DELJOLD, DELJNEW, SOLCRIT
  INTEGER :: I, J, K, L, LAY, M, ME
  REAL    :: SOURCET, NORM

  DELJDOT = 0.0
  DELJOLD = 0.0
  DELJNEW = 0.0
  NORM = 0.0
  DO LAY = 1, NLAY
    DO K = IXP(1,LAY), IXP(2,LAY)
      DO J = 1, NLM
        L = LOFJ(J)
        IF (L == 0) THEN
          SOURCET = CONSTSOURCE(J,K) + SHRADIANCE(J,K)*ALBEDOSC(LAY)
        ELSE
          SOURCET = CONSTSOURCE(J,K) &
                 + SHRADIANCE(J,K)*ALBEDOSC(LAY)*LEGENSC(L,LAY)/(2*L+1)
        ENDIF
        DELJDOT = DELJDOT + (SOURCET-SOURCE(J,K))*DELSOURCE(J,K)
        DELJOLD = DELJOLD + DELSOURCE(J,K)**2
        DELSOURCE(J,K) = SOURCET - SOURCE(J,K)
        DELJNEW = DELJNEW + (SOURCET-SOURCE(J,K))**2
        NORM = NORM + SOURCET**2
        SOURCE(J,K) = SOURCET
      ENDDO
    ENDDO
  ENDDO
  IF (NORM > 0.0) THEN
    SOLCRIT = SQRT(DELJNEW/NORM)
  ELSE
    SOLCRIT = 1.0
  ENDIF
END SUBROUTINE CALC_SOURCE
 



SUBROUTINE CALC_SOURCE2 (ML,MM, NLM, NLAY, NPTS, IXP, LOFJ, &
                         ALBEDOSC, LEGENSC, SHRADIANCE, CONSTSOURCE, &
                         SOURCE, DELSOURCE)
 ! Computes the source function (SOURCE) in spherical harmonic space 
 ! for all the grid levels in all the input layers.  
 ! The thermal source and/or solar pseudo-sources in CONSTSOURCE is
 ! added to the scattering source (computed from LEGENP and the spherical 
 ! harmonic expansion in SHRADIANCE).  This routine also computes the 
 ! new difference source function vector (DELSOURCE).  
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: ML, MM, NLM, NLAY, NPTS
  INTEGER, INTENT(IN) :: IXP(2,NLAY), LOFJ(NLM)
  REAL,    INTENT(IN) :: ALBEDOSC(NLAY), LEGENSC(1:ML,NLAY)
  REAL,    INTENT(IN) :: SHRADIANCE(NLM,NPTS), CONSTSOURCE(NLM,NPTS)
  REAL,    INTENT(INOUT) :: SOURCE(NLM,NPTS), DELSOURCE(NLM,NPTS)
  INTEGER :: J, K, L, LAY

  DO LAY = 1, NLAY
    DO K = IXP(1,LAY), IXP(2,LAY)
      SOURCE(1,K) = CONSTSOURCE(1,K) + SHRADIANCE(1,K)*ALBEDOSC(LAY)
      DO J = 2, NLM
        L = LOFJ(J)
        SOURCE(J,K) = CONSTSOURCE(J,K) &
                 + SHRADIANCE(J,K)*ALBEDOSC(LAY)*LEGENSC(L,LAY)/(2*L+1)
      ENDDO
    ENDDO
  ENDDO
END SUBROUTINE CALC_SOURCE2 




SUBROUTINE CALC_SRCINT (NLAY, NPTS, IXP, DTAUGSC, NMU, MU, SRCINT)
 ! Computes the source integration coefficients (SRCINT) from the sublayer
 ! delta optical depth array for the fixed grid.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY, NPTS, IXP(2,NLAY), NMU
  REAL,    INTENT(IN) :: MU(NMU)
  REAL,    INTENT(IN) :: DTAUGSC(NPTS)
  REAL,    INTENT(OUT) :: SRCINT(0:4,NMU,NPTS)
  INTEGER :: LAY, I, I1, I2, J, M
  DOUBLE PRECISION, ALLOCATABLE :: A(:,:), B(:,:), C(:,:), D(:,:), E(:)
  DOUBLE PRECISION, ALLOCATABLE :: T(:), T1(:), T3(:), T4(:), TR(:)

    ! Calculate and store the source function integration coefficients
  M=NMU/2
  ALLOCATE (A(4,M),B(4,M),C(4,M),D(4,M),E(M), T(M),T1(M),T3(M),T4(M), TR(M))
  DO LAY = 1, NLAY
    I1 = IXP(1,LAY)
    I2 = IXP(2,LAY)-1
    IF (I1 == I2) THEN   ! special case for layer with just two points
      T(:) = DTAUGSC(I1)/ABS(MU(1:M))
      TR(:) = EXP(-T(:))        ! sublayer transmission along ordinate
      IF (T(1) < 1.0E-4) THEN
        D(1,:) = 0.5*T(:) - (1.0D0/3)*T(:)**2
        D(2,:) = 0.5*T(:) - (1.0D0/6)*T(:)**2
      ELSE
        D(1,:) = (1-TR(:)*(1+T(:)))/T(:)
        D(2,:) = (T(:)-(1-TR(:)))/T(:)
      ENDIF
      SRCINT(0,1:M,I1) = TR(:)
      SRCINT(0,M+1:NMU,I1) = TR(:)
      SRCINT(1,1:M,I1) = D(2,:)
      SRCINT(2,1:M,I1) = D(1,:)
      SRCINT(1,M+1:NMU,I1) = D(1,:)
      SRCINT(2,M+1:NMU,I1) = D(2,:)
    ELSE
      DO I = I1, I2
        A(:,:)=0.0 ; B(:,:)=0.0 ; C(:,:)=0.0 ; D(:,:)=0.0
        T3(:) = DTAUGSC(I)/ABS(MU(1:M))
        IF (I == I1) THEN
          T4(:) = (DTAUGSC(I)+DTAUGSC(I+1))/ABS(MU(1:M))
          E=T3*T4      ; B(2,:)=1/E ; C(2,:)=-(T3+T4)/E ; D(2,:)=1.0
          E=T3*(T3-T4) ; B(3,:)=1/E ; C(3,:)=-T4/E      ; D(3,:)=0.0
          E=T4*(T4-T3) ; B(4,:)=1/E ; C(4,:)=-T3/E      ; D(4,:)=0.0
        ELSE IF (I < I2) THEN
          T1(:) = -DTAUGSC(I-1)/ABS(MU(1:M))
          T4(:) = (DTAUGSC(I)+DTAUGSC(I+1))/ABS(MU(1:M))
          E=T1*(T1-T3)*(T1-T4) ; A(1,:)=1/E
          B(1,:)=-(T3+T4)/E ; C(1,:)=T3*T4/E ; D(1,:)=0.0
          E=-T1*T3*T4 ; A(2,:)=1/E 
          B(2,:)=-(T1+T3+T4)/E ; C(2,:)=(T1*T3+T1*T4+T3*T4)/E ; D(2,:)=1.0
          E=T3*(T3-T1)*(T3-T4) ; A(3,:)=1/E
          B(3,:)=-(T1+T4)/E ; C(3,:)=T1*T4/E ; D(3,:)=0.0
          E=T4*(T4-T1)*(T4-T3) ; A(4,:)=1/E
          B(4,:)=-(T1+T3)/E ; C(4,:)=T1*T3/E ; D(4,:)=0.0
        ELSE
          T1(:) = -DTAUGSC(I-1)/ABS(MU(1:M))
          E=T1*(T1-T3) ; B(1,:)=1/E ; C(1,:)=-T3/E      ; D(1,:)=0.0
          E=T1*T3      ; B(2,:)=1/E ; C(2,:)=-(T1+T3)/E ; D(2,:)=1.0
          E=T3*(T3-T1) ; B(3,:)=1/E ; C(3,:)=-T1/E      ; D(3,:)=0.0
        ENDIF
        T(:) = T3(:)
        TR(:) = EXP(-T(:))
        SRCINT(0,1:M,I) = TR(:)
        SRCINT(0,M+1:NMU,I) = TR(:)
        DO J = 1, 4
          SRCINT(J,1:M,I) = A(J,:)*(6*TR+T**3-3*T**2+6*T-6) &
                          + B(J,:)*(-2*TR+T**2-2*T+2) &
                          + C(J,:)*(TR+T-1) + D(J,:)*(1-TR)
          SRCINT(J,M+1:NMU,I) = A(J,:)*(6-TR*(T**3+3*T**2+6*T+6)) &
                              + B(J,:)*(2-TR*(T**2+2*T+2)) &
                              + C(J,:)*(1-TR*(T+1)) + D(J,:)*(1-TR)
        ENDDO
      ENDDO
    ENDIF
  ENDDO
  DEALLOCATE (A, B, C, D, E, T, T1, T3, T4, TR)
END SUBROUTINE CALC_SRCINT



SUBROUTINE PATH_INTEG (NLAY, NPTS, IXP, &
                       NMU, NPHI0MAX, NPHI0, &
                       SRCTYPE, DIRFLUXSFC, SOLARMU, SKYRAD, &
                       SFCTYPE, SFCPARMS, SFCTEMP, WAVELEN, WAVENO, &
                       MU, PHI, WTDO, SRCINT, &
                       DOSOURCE, RADIANCE)
 ! Performs the path integrations of the source function (DOSOURCE) along
 ! the discrete ordinates to calculate the radiances.  The integrations 
 ! are done from the top boundary (with no reflection) down to the bottom 
 ! boundary for the downwelling angles, and then up from the bottom to the
 ! top after computing the surface reflection/emission.  The SRCINT 
 ! coefficient array must be input. 
 ! The discrete ordinates in MU, PHI must be a complete set with the
 ! downwelling (mu<0) angles first and downwelling and upwelling matching
 ! [MU(J) = -MU(NMU/2+J)].
 ! The surface reflection is handled differently for Lambertian surfaces 
 ! and more general surfaces specified by bidirectional reflection 
 ! distribution functions (BRDF).  For the general BRDF, the bottom 
 ! boundary radiance must be computed for each upwelling ordinate, while 
 ! for the Lambertian surface the boundary radiances can be computed just 
 ! once, since they are isotropic.  
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY, NPTS, IXP(2,NLAY)
  INTEGER, INTENT(IN) :: NMU, NPHI0MAX, NPHI0(NMU)
  REAL,    INTENT(IN) :: DIRFLUXSFC, SOLARMU, SKYRAD
  REAL,    INTENT(IN) :: SFCPARMS(*), SFCTEMP, WAVELEN, WAVENO(2)
  REAL,    INTENT(IN) :: MU(NMU), PHI(NPHI0MAX,NMU), WTDO(NMU,NPHI0MAX)
  REAL,    INTENT(IN) :: SRCINT(0:4,NMU,NPTS)
  REAL,    INTENT(IN) :: DOSOURCE(NPHI0MAX,NMU,NPTS)
  REAL,    INTENT(OUT) :: RADIANCE(NPHI0MAX,NMU,NPTS)
  CHARACTER(LEN=1), INTENT(IN) :: SRCTYPE, SFCTYPE
  INTEGER :: LAY, I, I1, I2, J, K, IPHI, IMU, M, N
  REAL    :: PI, TEMP, SFCPLANCK, PLANCKFUNC, ALB
  REAL    :: FLUXDNSFC, TOPRAD, BOTRAD

!$TAF INIT tape_path_integ = MEMORY
  PI = ACOS(-1.0)

  ! Make the isotropic radiances for the top boundary
  IF (SRCTYPE .EQ. 'T') THEN
    TOPRAD = PLANCKFUNC(WAVELEN,WAVENO,SKYRAD)
  ELSE
    TOPRAD = SKYRAD
  ENDIF
!$TAF STORE RADIANCE = tape_path_integ
  RADIANCE(:,1:NMU/2,IXP(1,1)) = TOPRAD
  FLUXDNSFC = 0.0


  ! First integrate the downwelling zenith angles (these angles must be first)
  ! For downwelling ordinates integrate source function from the top down
  !   Integrations are performed over the vector of azimuths
  DO IMU = 1, NMU/2
    N = NPHI0(IMU)
    DO LAY = 1, NLAY
      I1 = IXP(1,LAY)
      I2 = IXP(2,LAY)-1
      IF (I1 == I2) THEN
!$TAF STORE RADIANCE = tape_path_integ
        RADIANCE(1:N,IMU,I1+1) = SRCINT(0,IMU,I1)*RADIANCE(1:N,IMU,I1) &
                               + SRCINT(1,IMU,I1)*DOSOURCE(1:N,IMU,I1) &
                               + SRCINT(2,IMU,I1)*DOSOURCE(1:N,IMU,I1+1)
      ELSE
!$TAF STORE RADIANCE = tape_path_integ
        RADIANCE(1:N,IMU,I1+1) = SRCINT(0,IMU,I1)*RADIANCE(1:N,IMU,I1) &
                               + SRCINT(2,IMU,I1)*DOSOURCE(1:N,IMU,I1) &
                               + SRCINT(3,IMU,I1)*DOSOURCE(1:N,IMU,I1+1) &
                               + SRCINT(4,IMU,I1)*DOSOURCE(1:N,IMU,I1+2)
        DO I = I1+1, I2-1
!$TAF STORE RADIANCE = tape_path_integ
          RADIANCE(1:N,IMU,I+1) = SRCINT(0,IMU,I)*RADIANCE(1:N,IMU,I) &
                                + SRCINT(1,IMU,I)*DOSOURCE(1:N,IMU,I-1) &
                                + SRCINT(2,IMU,I)*DOSOURCE(1:N,IMU,I) &
                                + SRCINT(3,IMU,I)*DOSOURCE(1:N,IMU,I+1) &
                                + SRCINT(4,IMU,I)*DOSOURCE(1:N,IMU,I+2)
        ENDDO
!$TAF STORE RADIANCE = tape_path_integ
        RADIANCE(1:N,IMU,I2+1) = SRCINT(0,IMU,I2)*RADIANCE(1:N,IMU,I2) &
                               + SRCINT(1,IMU,I2)*DOSOURCE(1:N,IMU,I2-1) &
                               + SRCINT(2,IMU,I2)*DOSOURCE(1:N,IMU,I2) &
                               + SRCINT(3,IMU,I2)*DOSOURCE(1:N,IMU,I2+1)
      ENDIF
!$TAF STORE RADIANCE = tape_path_integ
      IF (LAY<NLAY) RADIANCE(:,IMU,IXP(1,LAY+1)) = RADIANCE(:,IMU,IXP(2,LAY))
    ENDDO

     ! Integrate over the DO radiances to get the downwelling surface flux
    DO K = 1, NPHI0(IMU)
      FLUXDNSFC = FLUXDNSFC &
                + ABS(MU(IMU))*WTDO(IMU,K)*RADIANCE(K,IMU,IXP(2,NLAY))
    ENDDO
  ENDDO
 
    ! Now do all the surface reflection stuff
  IF (SRCTYPE /= 'S') THEN
    SFCPLANCK = PLANCKFUNC(WAVELEN,WAVENO,SFCTEMP)
  ELSE
    SFCPLANCK = 0.0
  ENDIF

  ! For a Lambertian surface calculate the isotropic surface radiance
  !   from the thermal emission or direct reflection and the reflected 
  !   downwelling flux at the surface.
  IF (SFCTYPE == 'L') THEN
    ALB = SFCPARMS(1)
    IF (SRCTYPE == 'T') THEN
      BOTRAD = (1.0-ALB)*SFCPLANCK
    ELSE IF (SRCTYPE == 'S') THEN
      BOTRAD = ALB*DIRFLUXSFC/PI
    ELSE IF (SRCTYPE .EQ. 'B') THEN
      BOTRAD = (1.0-ALB)*SFCPLANCK + ALB*DIRFLUXSFC/PI
    ENDIF
!$TAF STORE RADIANCE = tape_path_integ
    RADIANCE(:,NMU/2+1:NMU,IXP(2,NLAY)) = BOTRAD + FLUXDNSFC*ALB/PI
  ELSE
    ! If not a Lambertian surface, compute the radiance for each of the
    !   upwelling ordinates by integrating the BRDF over the stored 
    !   downwelling radiances.
    DO IMU = NMU/2+1, NMU
      DO IPHI = 1, NPHI0(IMU)
!$TAF STORE RADIANCE = tape_path_integ
        CALL VARIABLE_BRDF_SURFACE (NMU, NPHI0MAX, NPHI0, MU, PHI, WTDO, &
                                    MU(IMU), PHI(IPHI,IMU), &
                                    SRCTYPE, SOLARMU, DIRFLUXSFC, &
                                    SFCTYPE, SFCPARMS, SFCPLANCK, &
                                    RADIANCE(:,:,IXP(2,NLAY)), &
                                    RADIANCE(IPHI,IMU,IXP(2,NLAY)))
      ENDDO
    ENDDO
  ENDIF


  ! Then integrate the upwelling zenith angles
      ! For upwelling ordinates integrate source function from the bottom up
  DO IMU = NMU/2+1, NMU
    N = NPHI0(IMU)
    DO LAY = NLAY, 1, -1
      I1 = IXP(1,LAY)
      I2 = IXP(2,LAY)-1
      IF (I1 == I2) THEN
!$TAF STORE RADIANCE = tape_path_integ
        RADIANCE(1:N,IMU,I1) = SRCINT(0,IMU,I1)*RADIANCE(1:N,IMU,I1+1) &
                             + SRCINT(1,IMU,I1)*DOSOURCE(1:N,IMU,I1) &
                             + SRCINT(2,IMU,I1)*DOSOURCE(1:N,IMU,I1+1)
      ELSE
!$TAF STORE RADIANCE = tape_path_integ
        RADIANCE(1:N,IMU,I2) = SRCINT(0,IMU,I2)*RADIANCE(1:N,IMU,I2+1) &
                             + SRCINT(1,IMU,I2)*DOSOURCE(1:N,IMU,I2-1) &
                             + SRCINT(2,IMU,I2)*DOSOURCE(1:N,IMU,I2) &
                             + SRCINT(3,IMU,I2)*DOSOURCE(1:N,IMU,I2+1)
        DO I = I2-1, I1+1, -1
!$TAF STORE RADIANCE = tape_path_integ
          RADIANCE(1:N,IMU,I) = SRCINT(0,IMU,I)*RADIANCE(1:N,IMU,I+1) &
                              + SRCINT(1,IMU,I)*DOSOURCE(1:N,IMU,I-1) &
                              + SRCINT(2,IMU,I)*DOSOURCE(1:N,IMU,I) &
                              + SRCINT(3,IMU,I)*DOSOURCE(1:N,IMU,I+1) &
                              + SRCINT(4,IMU,I)*DOSOURCE(1:N,IMU,I+2)
        ENDDO
!$TAF STORE RADIANCE = tape_path_integ
        RADIANCE(1:N,IMU,I1) = SRCINT(0,IMU,I1)*RADIANCE(1:N,IMU,I1+1) &
                             + SRCINT(2,IMU,I1)*DOSOURCE(1:N,IMU,I1) &
                             + SRCINT(3,IMU,I1)*DOSOURCE(1:N,IMU,I1+1) &
                             + SRCINT(4,IMU,I1)*DOSOURCE(1:N,IMU,I1+2)
      ENDIF
!$TAF STORE RADIANCE = tape_path_integ
      IF (LAY>1) RADIANCE(:,IMU,IXP(2,LAY-1)) = RADIANCE(:,IMU,IXP(1,LAY))
    ENDDO
  ENDDO

END SUBROUTINE PATH_INTEG




SUBROUTINE DELTA_SCALE (NLAY, ML, TAUP, ALBEDOP, LEGENP, &
                        TAUPSC, ALBEDOSC, LEGENSC)
 ! Delta scales the layer optical depth, albedo, and Legendre terms;
 ! only the 1 to ML LEGEN terms are scaled.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY, ML
  REAL,    INTENT(IN) ::  TAUP(NLAY), ALBEDOP(NLAY), LEGENP(1:ML+1,NLAY)
  REAL,    INTENT(OUT) ::  TAUPSC(NLAY), ALBEDOSC(NLAY), LEGENSC(1:ML,NLAY)
  INTEGER :: I, L
  REAL    :: F

  DO I = 1, NLAY
    F = LEGENP(ML+1,I)/(2*(ML+1)+1)
    DO L = 1, ML
      LEGENSC(L,I) = (2*L+1)*(LEGENP(L,I)/(2*L+1) - F)/(1-F)
    ENDDO
    TAUPSC(I) = (1.0-ALBEDOP(I)*F)*TAUP(I)
    ALBEDOSC(I) = (1.0-F)*ALBEDOP(I)/(1.0-ALBEDOP(I)*F)
  ENDDO
END SUBROUTINE DELTA_SCALE



SUBROUTINE MAKE_TAU_GRID (NLAY, TAUPSC, ALBEDOSC, MAXDELTAU, MINDELTAUSPLIT, &
                          NPTS, TAUGSC, IXP, IXG)
 ! Makes the initial optical depth grid, TAUGSC, of increasing values
 ! (representing decreasing altitudes in the atmosphere). TAUGSC(1) is zero.
 ! The "base" tau grid is the input property grid boundaries and even 
 ! subdivision of each of these layers so that the sublayers are less 
 ! than MAXDELTAU scattering optical depth.   All layers with scaled
 ! optical depth greater than MINDELTAUSPLIT are split into two sublayers.
 ! The IXP index array points to the top and bottom TAUGSC levels for each 
 ! property layer. The IXG index array points to the source and 
 ! radiance arrays for each TAUGSC level.
  IMPLICIT NONE
  INTEGER, INTENT(IN)  :: NLAY
  REAL,    INTENT(IN)  :: TAUPSC(NLAY), ALBEDOSC(NLAY)
  REAL,    INTENT(IN)  :: MAXDELTAU, MINDELTAUSPLIT
  REAL,    INTENT(OUT) :: TAUGSC(*)
  INTEGER, INTENT(OUT) :: NPTS, IXP(2,NLAY), IXG(*)
  INTEGER :: I, L, N
  REAL    :: TAU

  NPTS = 0
  TAU = 0.0
  DO L = 1, NLAY
    NPTS = NPTS + 1
    TAUGSC(NPTS) = TAU
    IXP(1,L) = NPTS
    IF (TAUPSC(L) > MINDELTAUSPLIT) THEN
      N = MAX(1,INT(ALBEDOSC(L)*TAUPSC(L)/MAXDELTAU))
    ELSE
      N = 0
    ENDIF
    DO I = 1, N
      NPTS = NPTS + 1
      TAUGSC(NPTS) = TAUGSC(IXP(1,L)) + TAUPSC(L)*FLOAT(I)/(N+1)
    ENDDO
    NPTS = NPTS + 1
    IXP(2,L) = NPTS
    TAUGSC(NPTS) = TAUGSC(IXP(1,L)) + TAUPSC(L)
    TAU = TAUGSC(NPTS)
  ENDDO
  IXG(1:NPTS) = (/ (I, I=1,NPTS) /)
END SUBROUTINE MAKE_TAU_GRID




SUBROUTINE PREPARE_SOURCE (NLAY, TAUPSC, ALBEDOSC, TEMPP, LEGENSC, &
                           SRCTYPE, ML, MM, NLM, NPTS, TAUGSC, IXP, IXG, &
                           SOLARFLUX, SOLARMU, YLMSUN, WAVELEN, WAVENO, &
                           CONSTSOURCE)
 ! Prepares the constant part of the source function for each level.
 ! This is either or both thermal emission or the single scattering
 ! solar pseudo-source.  Only those levels (points) with CONSTSOURCE(1,:)
 ! less than zero are changed, which allows this to be run for each time
 ! adaptive grid points are added.
 ! The thermal emission source is \sqrt{4\pi}(1-\omega)B(T), where the
 ! temperature T is assumed to vary linearly in optical depth.
 ! The solar source is 
 !   S_0/\mu_0 e^{-\tau/\mu_0} Y_{lm}(\mu_0,\phi_0) {\omega \chi_l \over 2l+1}
 ! where S_0 is the solar flux on a horizontal plane, \chi_l are the Legendre
 ! coefficients, and \tau is the optical depth at the level.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY, ML, MM, NLM, NPTS
  INTEGER, INTENT(IN) :: IXP(2,NLAY), IXG(NPTS)
  REAL,    INTENT(IN) :: SOLARFLUX, SOLARMU, YLMSUN(NLM), WAVELEN, WAVENO(2)
  REAL,    INTENT(IN) :: TAUPSC(NLAY), ALBEDOSC(NLAY), LEGENSC(1:ML,NLAY)
  REAL,    INTENT(IN) :: TEMPP(NLAY+1), TAUGSC(NPTS)
  REAL,    INTENT(INOUT) :: CONSTSOURCE(NLM,NPTS)
  CHARACTER(LEN=1), INTENT(IN) :: SRCTYPE
  INTEGER :: I, J, K, L, LAY, ME, M, LOFJ(NLM)
  REAL    :: S0, C, TAU1, U, TEMP, BB, D, PLANCKFUNC

  S0 = SOLARFLUX/ABS(SOLARMU)
  C = SQRT(4.0*ACOS(-1.0))
  ! Make the l index as a function of SH term (J)
  J = 0
  DO L = 0, ML
    ME = MIN(L,MM)
    DO M = 0, ME
      J = J + 1
      LOFJ(J) = L
    ENDDO
  ENDDO

  DO LAY = 1, NLAY
    TAU1 = TAUGSC(IXP(1,LAY))
    DO I = IXP(1,LAY), IXP(2,LAY)
      K = IXG(I)
      IF (CONSTSOURCE(1,K) < 0) THEN
        CONSTSOURCE(:,K) = 0.0
        IF (SRCTYPE /= 'S') THEN
          IF (TAUPSC(LAY) > 0.0) THEN
            U = (TAUGSC(I)-TAU1)/TAUPSC(LAY)
          ELSE
            U = (I-IXP(1,LAY))/FLOAT(IXP(2,LAY)-IXP(1,LAY))
          ENDIF
          TEMP = TEMPP(LAY) + U*(TEMPP(LAY+1)-TEMPP(LAY))
          CONSTSOURCE(1,K) = C*(1.0-ALBEDOSC(LAY)) &
                            *PLANCKFUNC(WAVELEN,WAVENO,TEMP)
        ENDIF
        IF (SRCTYPE /= 'T') THEN
          D = S0*EXP(-TAUGSC(I)/ABS(SOLARMU))
          CONSTSOURCE(1,K) = CONSTSOURCE(1,K) + D*YLMSUN(1)*ALBEDOSC(LAY)
          DO J = 2, NLM
            L = LOFJ(J)
            CONSTSOURCE(J,K) = CONSTSOURCE(J,K) &
                           + D*YLMSUN(J)*ALBEDOSC(LAY)*LEGENSC(L,LAY)/(2*L+1)
          ENDDO
        ENDIF
      ENDIF
    ENDDO
  ENDDO
END SUBROUTINE PREPARE_SOURCE


FUNCTION PLANCKFUNC (WAVELEN, WAVENO, TEMP)
 ! Calculates the Planck blackbody radiance. If WAVELEN<=0 then doing a 
 ! spectral band integration and the Planck blackbody radiance in 
 ! [Watts /(meter^2 ster)] over a wavenumber range [cm^-1] (WAVENO) is 
 ! returned.  Otherwise, the Planck blackbody radiance in 
 ! [Watts /(meter^2 ster micron)] for a temperature in [Kelvins] at 
 ! a wavelength (WAVELEN) in [microns] is returned.
  IMPLICIT NONE
  REAL, INTENT(IN) :: WAVELEN, WAVENO(2), TEMP
  REAL :: PLANCKFUNC
  DOUBLE PRECISION :: X1, X2, F

  IF (TEMP > 0.0) THEN
    IF (WAVELEN>0) THEN
      PLANCKFUNC = 1.1911E8 / WAVELEN**5 / (EXP(1.4388E4/(WAVELEN*TEMP)) - 1)
    ELSE
      X1 = 1.4388D0*WAVENO(1)/TEMP
      X2 = 1.4388D0*WAVENO(2)/TEMP
      CALL INTEGRATE_PLANCK (X1, X2, F)
      PLANCKFUNC = 1.1911D-8*(TEMP/1.4388D0)**4 *F
    ENDIF
  ELSE
    PLANCKFUNC = 0.0
  ENDIF
END FUNCTION PLANCKFUNC



SUBROUTINE MAKE_ANGLE_SET (NMU, NPHI, NCS, NPHI0MAX, ITYPE, &
                           NPHI0, MU, PHI, WTMU, WTDO, NANG)
 ! Make the set of angles for the discrete space representation.
 ! The number of mu's (cosine zenith angles) and maximum number of
 ! phi's (azimuth angles) is input.  The actual number of azimuthal 
 ! angles for each mu is output (NPHI0).  There are three types of
 ! discrete ordinate sets: ITYPE=1 is a gaussian grid, 2 is a reduced
 ! gaussian grid, and 3 is a reduced double gaussian set.
 ! If NCS=1 then only about half the azimuthal angles (from 0 to pi) 
 ! are used because the radiance is even in phi (cosine terms).  
 ! The output is the NMU mu values, the NPHI0 phi values for each mu,
 ! and the integration weight for each ordinate. The first NMU/2 mu 
 ! angles are the downwelling (mu<0) angles.  Also output is the
 ! total number of angles (NANG).
  IMPLICIT NONE
  INTEGER, INTENT(IN)  :: NMU, NPHI, NCS, NPHI0MAX, ITYPE
  INTEGER, INTENT(OUT) :: NPHI0(NMU), NANG
  REAL,    INTENT(OUT) :: MU(NMU), WTMU(NMU)
  REAL,    INTENT(OUT) :: PHI(NPHI0MAX,NMU), WTDO(NMU,NPHI0MAX)
  INTEGER :: J, K, MM
  REAL    :: DELPHI

  MM = MAX(0,INT(NPHI/2)-1)
  NANG = 0
  IF (ITYPE .GE. 1 .OR. ITYPE .LE. 3) THEN
    IF (ITYPE .LE. 2) THEN
      CALL GAUSQUADS (NMU, MU, WTMU)
    ELSE IF (ITYPE .EQ. 3) THEN
      CALL DGAUSQUADS (NMU, MU, WTMU)
    ENDIF
    DO J = 1, NMU
      IF (NCS .EQ. 1) THEN
        IF (NPHI0MAX .NE. INT((NPHI+2)/2)) STOP 'MAKE_ANGLE_SET: bad NPHI0MAX'
        IF (ITYPE .EQ. 1) THEN
          NPHI0(J) = NPHI0MAX
        ELSE
        ! For the reduced gaussian set, make the smaller NPHI0 values
          NPHI0(J) = INT(0.9+1+(NPHI0MAX-1)*SQRT(1-MU(J)**2))
        ENDIF
        ! Compute the azimuth angles and weights
        DELPHI = ACOS(-1.0)/MAX(1,NPHI0(J)-1)
        DO K = 1, NPHI0(J)
          PHI(K,J) = (K-1)*DELPHI
          IF ((K.EQ.1 .OR. K.EQ.NPHI0(J)) .AND. NPHI0(J).NE.1) THEN
            WTDO(J,K) = DELPHI*WTMU(J)
          ELSE
            WTDO(J,K) = 2.0*DELPHI*WTMU(J)
          ENDIF
        ENDDO
      ELSE
        IF (NPHI0MAX .NE. NPHI) STOP 'MAKE_ANGLE_SET: bad NPHI0MAX'
        IF (ITYPE .EQ. 1) THEN
          NPHI0(J) = NPHI0MAX
        ELSE
          NPHI0(J) = INT(0.9+NPHI0MAX*SQRT(1-MU(J)**2))
        ENDIF
        DELPHI = 2.0*ACOS(-1.0)/NPHI0(J)
        DO K = 1, NPHI0(J)
          PHI(K,J) = (K-1)*DELPHI
          WTDO(J,K) = DELPHI*WTMU(J)
        ENDDO
      ENDIF
      NANG = NANG + NPHI0(J)
    ENDDO
        
  ELSE
    STOP 'MAKE_ANGLE_SET: invalid discrete ordinate type'
  ENDIF
END SUBROUTINE MAKE_ANGLE_SET




SUBROUTINE MAKE_SH_DO_COEF (ML, MM, NLM, NMU, NPHI0, &
                            NPHI0MAX, MU, PHI, WTMU, WTDO, &
                            CMU1, CMU2, CPHI1, CPHI2)
 ! Makes the transformation coefficients for the spherical harmonic
 ! transform.  The main coefficients are output in four arrays: 1 is for
 ! the SH_TO_DO forward transform, 2 is for the DO_TO_SH back transform,
 ! which contains the discrete angle integration weights.
 ! The coefficients are divided up into the mu dependent set CMUn 
 ! (function of l, m, mu_j), and the phi dependent set CPHIn 
 ! (function of m, phi_k) for each mu_j.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: ML, MM, NLM, NMU, NPHI0(NMU), NPHI0MAX
  REAL,    INTENT(IN) :: MU(NMU), WTMU(NMU)
  REAL,    INTENT(IN) :: PHI(NPHI0MAX,NMU), WTDO(NMU,NPHI0MAX)
  REAL,    INTENT(OUT) :: CMU1(NLM,NMU), CMU2(NMU,NLM)
  REAL,    INTENT(OUT) :: CPHI1(0:MM,NPHI0MAX,NMU), CPHI2(NPHI0MAX,0:MM,NMU)
  INTEGER :: I, K, M
  REAL    :: W

  ! Make the to and from associate Legendre coefficients       
  DO I = 1, NMU
    CALL YLMALL (MU(I), 0.0, ML, MM, -1, CMU1(1,I))
  ENDDO
  DO I = 1, NMU
    CMU2(I,1:NLM) = CMU1(1:NLM,I)*WTMU(I)
  ENDDO

  ! Make the to and from Fourier coefficients for each mu
  DO I = 1, NMU
    ! Make the DFT coefficient
    W = 1.0/WTMU(I)
    DO K = 1, NPHI0(I)
      DO M = 0, MM
        CPHI1(M,K,I)  = COS(M*PHI(K,I))
        CPHI2(K,M,I)  = CPHI1(M,K,I)*WTDO(I,K)*W
      ENDDO
    ENDDO
  ENDDO
END SUBROUTINE MAKE_SH_DO_COEF
 



SUBROUTINE SH_TO_DO (NPTS, ML, MM, NLM, NMU, NPHI0MAX, NPHI0, &
                     CMU1, CPHI1, INDATA, OUTDATA)
 ! Transforms the input data from spherical harmonic space to discrete 
 ! ordinate space for all points in the array.  Successive transforms are 
 ! done in zenith and then azimuth angle.  The Fourier transform in 
 ! azimuth is done using a matrix multiply DFT.  The number of floating 
 ! point operations for the DFT is 2*Nmu*Nphi0*Nm.  The number of floating 
 ! point operations for the zenith angle transform is 2*Nmu*Nlm.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NPTS, ML, MM, NLM, NMU, NPHI0MAX, NPHI0(NMU)
  REAL,    INTENT(IN) :: INDATA(NLM,NPTS)
  REAL,    INTENT(IN) :: CMU1(NLM,NMU), CPHI1(0:MM,NPHI0MAX,NMU)
  REAL,    INTENT(OUT) :: OUTDATA(NPHI0MAX,NMU,NPTS)
  INTEGER :: I, J, L, M, ME, IMU, IPHI
  INTEGER :: MOFJ(NLM)
  REAL, ALLOCATABLE :: S(:,:)

  ! Make the M for each J array
  J = 0
  DO L = 0, ML
    DO M = 0, MIN(L,MM)
      J = J + 1
      MOFJ(J) = M
    ENDDO
  ENDDO

  ! Do the transform for each grid point
  ALLOCATE (S(0:MM,NPTS))
  DO I = 1, NPTS
    DO IMU = 1, NMU
      ! Figure the max Fourier azimuthal mode we can do for this Nphi0
      ME = MAX(0,MIN(NPHI0(IMU)-2,MOFJ(NLM)))
      ! First do Legendre transform by summing over l for each m.
      S(:,I) = 0.0
      DO J = 1, NLM
        M = MOFJ(J)
        S(M,I) = S(M,I) + CMU1(J,IMU)*INDATA(J,I)
      ENDDO
      ! Then do the discrete Fourier transform from m to phi for each mu
      DO IPHI = 1, NPHI0(IMU)
        OUTDATA(IPHI,IMU,I) = SUM( CPHI1(0:ME,IPHI,IMU)*S(0:ME,I) )
      ENDDO
    ENDDO
  ENDDO
  DEALLOCATE (S)
END SUBROUTINE SH_TO_DO



SUBROUTINE DO_TO_SH (NPTS, ML, MM, NLM, NMU, NPHI0MAX, NPHI0, &
                     CMU2, CPHI2, INDATA, OUTDATA)
 ! Transforms the input field from discrete ordinate space to spherical 
 ! harmonic space for all the points.  (See SH_TO_DO for more).
 ! The SELECTPNTS logical array indicates whether a point should be 
 ! transformed or to use the previous points (for the input layer interfaces).
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NPTS, ML, MM, NLM, NMU, NPHI0MAX, NPHI0(NMU)
  REAL,    INTENT(IN) :: INDATA(NPHI0MAX,NMU,NPTS)
  REAL,    INTENT(IN) :: CMU2(NMU,NLM), CPHI2(NPHI0MAX,0:MM,NMU)
  REAL,    INTENT(OUT) :: OUTDATA(NLM,NPTS)
  INTEGER :: I, IMU, IPHI, J, L, M, ME, N
  REAL    :: PI, DELPHI
  INTEGER :: MOFJ(NLM)
  REAL, ALLOCATABLE :: S(:,:)

  PI = ACOS(-1.0)
  ! Make the M for each J array
  J = 0
  DO L = 0, ML
    DO M = 0, MIN(L,MM)
      J = J + 1
      MOFJ(J) = M
    ENDDO
  ENDDO

  ! Do the transform for each grid point
  ALLOCATE (S(0:NPHI0MAX-1,NPTS))
  DO I = 1, NPTS
    OUTDATA(:,I) = 0.0
    DO IMU = 1, NMU
      N = NPHI0(IMU)
      ! Figure the max Fourier azimuthal mode we can do for this Nphi0
      ME = MAX(0,MIN(N-2,MOFJ(NLM)))
      ! First do the discrete Fourier transform from phi to m for each mu
      S(:,I) = 0.0
      DO M = 0, ME
        S(M,I) = SUM( CPHI2(1:N,M,IMU)*INDATA(1:N,IMU,I) )
      ENDDO
      ! Then do Legendre transform by adding to output for each l, m
      DO J = 1, NLM
        M = MOFJ(J)
        OUTDATA(J,I) = OUTDATA(J,I) + CMU2(IMU,J)*S(M,I)
      ENDDO
    ENDDO
  ENDDO
  DEALLOCATE (S)
END SUBROUTINE DO_TO_SH




SUBROUTINE COMPUTE_SOURCE (ML,MM, NLM, NLAY, NPTS, IXP, IXG, &
                           ALBEDOSC, LEGENSC, &
                           SHRADIANCE, CONSTSOURCE, SOURCE, &
                           OLDNPTS, DELSOURCE, &
                           DELJDOT, DELJOLD, DELJNEW, SOLCRIT)
 ! Computes the source function (SOURCE) in spherical harmonic space 
 ! for all the grid levels in all the input layers.  
 ! The thermal source and/or solar pseudo-sources in CONSTSOURCE is
 ! added to the scattering source (computed from LEGENP and the spherical 
 ! harmonic expansion in SHRADIANCE).
 ! To save memory this routine also computes some things for the sequence
 ! acceleration: the dot products of the difference in source function 
 ! between successive iterations and the new difference source function 
 ! vector (DELSOURCE).  
 ! Computes the new solution criterion (SOLCRIT, the RMS difference
 ! in succesive source function fields normalized by the RMS of the field
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: ML, MM, NLM, NLAY, NPTS, OLDNPTS
  INTEGER, INTENT(IN) :: IXP(2,NLAY), IXG(NPTS)
  REAL,    INTENT(IN) :: ALBEDOSC(NLAY), LEGENSC(1:ML,NLAY)
  REAL,    INTENT(IN) :: SHRADIANCE(NLM,NPTS), CONSTSOURCE(NLM,NPTS)
  REAL,    INTENT(INOUT) :: SOURCE(NLM,NPTS), DELSOURCE(NLM,OLDNPTS)
  REAL,    INTENT(OUT) :: DELJDOT, DELJOLD, DELJNEW, SOLCRIT
  INTEGER :: LOFJ(NLM), I, J, K, L, LAY, M, ME
  REAL    :: SOURCET, NORM

  ! Make the l index as a function of SH term (J)
  J = 0
  DO L = 0, ML
    ME = MIN(L,MM)
    DO M = 0, ME
      J = J + 1
      LOFJ(J) = L
    ENDDO
  ENDDO

  
  DELJDOT = 0.0
  DELJOLD = 0.0
  DELJNEW = 0.0
  NORM = 0.0
  DO LAY = 1, NLAY
    DO I = IXP(1,LAY), IXP(2,LAY)
      K = IXG(I)
      DO J = 1, NLM
        L = LOFJ(J)
        IF (L == 0) THEN
          SOURCET = CONSTSOURCE(1,K) + SHRADIANCE(1,K)*ALBEDOSC(LAY)
        ELSE
          SOURCET = CONSTSOURCE(J,K) &
                  + SHRADIANCE(J,K)*ALBEDOSC(LAY)*LEGENSC(L,LAY)/(2*L+1)
        ENDIF
        IF (K <= OLDNPTS) THEN
          DELJDOT = DELJDOT + (SOURCET-SOURCE(J,K))*DELSOURCE(J,K)
          DELJOLD = DELJOLD + DELSOURCE(J,K)**2
          DELSOURCE(J,K) = SOURCET - SOURCE(J,K)
          DELJNEW = DELJNEW + (SOURCET-SOURCE(J,K))**2
          NORM = NORM + SOURCET**2
        ENDIF
        SOURCE(J,K) = SOURCET
      ENDDO
    ENDDO
  ENDDO
  IF (NORM > 0.0) THEN
    SOLCRIT = SQRT(DELJNEW/NORM)
  ELSE
    SOLCRIT = 1.0
  ENDIF
END SUBROUTINE COMPUTE_SOURCE
 




SUBROUTINE ACCELERATE_SOLUTION (NLM, NPTS, OLDNPTS, SOURCE, DELSOURCE, &
                               DELJDOT, DELJOLD, DELJNEW, A)
 ! Accelerates the successive order of scattering series.  Uses
 ! information about the ratio in lengths and angle between successive
 ! source function differences to decide how far along the current
 ! source function difference to extrapolate.  The acceleration 
 ! step size (A) is returned, and is also input to see if there was
 ! an acceleration the last iteration.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLM, NPTS, OLDNPTS
  REAL,    INTENT(IN) :: DELSOURCE(NLM,OLDNPTS)
  REAL,    INTENT(INOUT) :: SOURCE(NLM,NPTS)
  REAL,    INTENT(IN) :: DELJDOT, DELJOLD, DELJNEW
  REAL,    INTENT(INOUT) :: A
  INTEGER :: I, J, IS, ISD, NS, NSD, NSC
  REAL    :: R, THETA

  ! Accelerate if the grids are the same, didn't last time, and 
  !   things are converging. 
  IF (NPTS == OLDNPTS .AND. A == 0.0 .AND. DELJNEW < DELJOLD) THEN
    ! Compute the acceleration extrapolation factor and apply it.
    R = SQRT(DELJNEW/DELJOLD)
    THETA = ACOS(DELJDOT/SQRT(DELJOLD*DELJNEW))
    A = (1 - R*COS(THETA) + R**(1+0.5*3.14159/THETA)) &
         /(1 + R**2  - 2*R*COS(THETA))  - 1.0
    A = MIN(9.99,MAX(0.0,A))
    ! WRITE (*,'(1X,A,3(1X,F7.3))') '! Acceleration: ', A,R,THETA
    SOURCE(:,:) = SOURCE(:,:) + A*DELSOURCE(:,:)
  ELSE
    A = 0.0
  ENDIF
END SUBROUTINE ACCELERATE_SOLUTION




SUBROUTINE PATH_INTEGRATION (NLAY, NPTS, IXP, IXG, TAUGSC, &
                             NMU, NPHI0MAX, NPHI0, &
                             SRCTYPE, SOLARFLUX, SOLARMU, SKYRAD, &
                             SFCTYPE, SFCPARMS, SFCTEMP, WAVELEN, WAVENO, &
                             MU, PHI, WTDO, SRCINT, LASTNPTS, &
                             DOSOURCE, RADIANCE)
 ! Performs the path integrations through the medium specified by
 ! the optical depth (TAUGSC) and source function (DOSOURCE) in
 ! discrete ordinates.  The integrations are done from the top boundary
 ! (with no reflection) down to the bottom boundary for the downwelling 
 ! angles, and then up from the bottom to the top after computing the 
 ! surface reflection/emission.   
 ! The discrete ordinates in MU, PHI must be a complete set with the
 ! downwelling (mu<0) angles first and downwelling and upwelling matching
 ! [MU(J) = -MU(NMU/2+J)].
 ! The surface reflection is handled differently for Lambertian surfaces 
 ! and more general surfaces specified by bidirectional reflection 
 ! distribution functions (BRDF).  For the general BRDF, the bottom 
 ! boundary radiance must be computed for each upwelling ordinate, while 
 ! for the Lambertian surface the boundary radiances can be computed just 
 ! once, since they are isotropic.  
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY, NPTS, IXP(2,NLAY), IXG(NPTS)
  INTEGER, INTENT(IN) :: NMU, NPHI0MAX, NPHI0(NMU)
  INTEGER, INTENT(INOUT) :: LASTNPTS
  REAL,    INTENT(IN) :: SOLARFLUX, SOLARMU, SKYRAD
  REAL,    INTENT(IN) :: SFCPARMS(*), SFCTEMP, WAVELEN, WAVENO(2)
  REAL,    INTENT(IN) :: MU(NMU), PHI(NPHI0MAX,NMU), WTDO(NMU,NPHI0MAX)
  REAL,    INTENT(INOUT) :: SRCINT(0:4,NMU,NPTS)
  REAL,    INTENT(IN) :: TAUGSC(NPTS), DOSOURCE(NPHI0MAX,NMU,NPTS)
  REAL,    INTENT(OUT) :: RADIANCE(NPHI0MAX,NMU,NPTS)
  CHARACTER(LEN=1), INTENT(IN) :: SRCTYPE, SFCTYPE
 
  INTEGER :: LAY, I, I1, I2, J, K, IPHI, IMU, M, N
  REAL    :: PI, TEMP, SFCPLANCK, PLANCKFUNC, ALB
  REAL    :: DIRFLUXSFC, FLUXDNSFC, TOPRAD, BOTRAD
  DOUBLE PRECISION, ALLOCATABLE :: A(:,:), B(:,:), C(:,:), D(:,:), E(:)
  DOUBLE PRECISION, ALLOCATABLE :: T(:), T1(:), T3(:), T4(:), TR(:)

  PI = ACOS(-1.0)

  M=NMU/2
  
  IF (LASTNPTS /= NPTS) THEN
    ! Calculate and store the source function integration coefficients
    ALLOCATE (A(4,M),B(4,M),C(4,M),D(4,M),E(M), T(M),T1(M),T3(M),T4(M), TR(M))
    DO LAY = 1, NLAY
      I1 = IXP(1,LAY)
      I2 = IXP(2,LAY)-1
      IF (I1 == I2) THEN   ! special case for layer with just two points
        T(:) = (TAUGSC(I1+1) - TAUGSC(I1))/ABS(MU(1:M))
        TR(:) = EXP(-T(:))        ! sublayer transmission along ordinate
        IF (T(1) < 1.0E-4) THEN
          D(1,:) = 0.5*T(:) - (1.0D0/3)*T(:)**2
          D(2,:) = 0.5*T(:) - (1.0D0/6)*T(:)**2
        ELSE
          D(1,:) = (1-TR(:)*(1+T(:)))/T(:)
          D(2,:) = (T(:)-(1-TR(:)))/T(:)
        ENDIF
        SRCINT(0,1:M,I1) = TR(:)
        SRCINT(0,M+1:NMU,I1) = TR(:)
        SRCINT(1,1:M,I1) = D(2,:)
        SRCINT(2,1:M,I1) = D(1,:)
        SRCINT(1,M+1:NMU,I1) = D(1,:)
        SRCINT(2,M+1:NMU,I1) = D(2,:)
      ELSE
        DO I = I1, I2
          A(:,:)=0.0 ; B(:,:)=0.0 ; C(:,:)=0.0 ; D(:,:)=0.0
          T3(:) = (TAUGSC(I+1)-TAUGSC(I))/ABS(MU(1:M))
          IF (I == I1) THEN
            T4(:) = (TAUGSC(I+2)-TAUGSC(I))/ABS(MU(1:M))
            E=T3*T4      ; B(2,:)=1/E ; C(2,:)=-(T3+T4)/E ; D(2,:)=1.0
            E=T3*(T3-T4) ; B(3,:)=1/E ; C(3,:)=-T4/E      ; D(3,:)=0.0
            E=T4*(T4-T3) ; B(4,:)=1/E ; C(4,:)=-T3/E      ; D(4,:)=0.0
          ELSE IF (I < I2) THEN
            T1(:) = (TAUGSC(I-1)-TAUGSC(I))/ABS(MU(1:M))
            T4(:) = (TAUGSC(I+2)-TAUGSC(I))/ABS(MU(1:M))
            E=T1*(T1-T3)*(T1-T4) ; A(1,:)=1/E
            B(1,:)=-(T3+T4)/E ; C(1,:)=T3*T4/E ; D(1,:)=0.0
            E=-T1*T3*T4 ; A(2,:)=1/E 
            B(2,:)=-(T1+T3+T4)/E ; C(2,:)=(T1*T3+T1*T4+T3*T4)/E ; D(2,:)=1.0
            E=T3*(T3-T1)*(T3-T4) ; A(3,:)=1/E
            B(3,:)=-(T1+T4)/E ; C(3,:)=T1*T4/E ; D(3,:)=0.0
            E=T4*(T4-T1)*(T4-T3) ; A(4,:)=1/E
            B(4,:)=-(T1+T3)/E ; C(4,:)=T1*T3/E ; D(4,:)=0.0
          ELSE
            T1(:) = (TAUGSC(I-1)-TAUGSC(I))/ABS(MU(1:M))
            E=T1*(T1-T3) ; B(1,:)=1/E ; C(1,:)=-T3/E      ; D(1,:)=0.0
            E=T1*T3      ; B(2,:)=1/E ; C(2,:)=-(T1+T3)/E ; D(2,:)=1.0
            E=T3*(T3-T1) ; B(3,:)=1/E ; C(3,:)=-T1/E      ; D(3,:)=0.0
          ENDIF
          T(:) = T3(:)
          TR(:) = EXP(-T(:))
          SRCINT(0,1:M,I) = TR(:)
          SRCINT(0,M+1:NMU,I) = TR(:)
          DO J = 1, 4
            SRCINT(J,1:M,I) = A(J,:)*(6*TR+T**3-3*T**2+6*T-6) &
                            + B(J,:)*(-2*TR+T**2-2*T+2) &
                            + C(J,:)*(TR+T-1) + D(J,:)*(1-TR)
            SRCINT(J,M+1:NMU,I) = A(J,:)*(6-TR*(T**3+3*T**2+6*T+6)) &
                                + B(J,:)*(2-TR*(T**2+2*T+2)) &
                                + C(J,:)*(1-TR*(T+1)) + D(J,:)*(1-TR)
          ENDDO
        ENDDO
      ENDIF
    ENDDO
    DEALLOCATE (A, B, C, D, E, T, T1, T3, T4, TR)
  ENDIF


  ! Make the isotropic radiances for the top boundary
  IF (SRCTYPE .EQ. 'T') THEN
    TEMP = SKYRAD
    TOPRAD = PLANCKFUNC(WAVELEN,WAVENO,TEMP)
  ELSE
    TOPRAD = SKYRAD
  ENDIF
  RADIANCE(:,1:NMU/2,IXG(IXP(1,1))) = TOPRAD
  FLUXDNSFC = 0.0


  ! First integrate the downwelling zenith angles (these angles must be first)
  ! For downwelling ordinates integrate source function from the top down
  !   Integrations are performed over the vector of azimuths
  DO IMU = 1, NMU/2
    N = NPHI0(IMU)
    DO LAY = 1, NLAY
      I1 = IXP(1,LAY)
      I2 = IXP(2,LAY)-1
      IF (I1 == I2) THEN
        RADIANCE(1:N,IMU,IXG(I1+1)) = SRCINT(0,IMU,I1)*RADIANCE(1:N,IMU,IXG(I1)) &
                             + SRCINT(1,IMU,I1)*DOSOURCE(1:N,IMU,IXG(I1)) &
                             + SRCINT(2,IMU,I1)*DOSOURCE(1:N,IMU,IXG(I1+1))
      ELSE
        RADIANCE(1:N,IMU,IXG(I1+1)) = SRCINT(0,IMU,I1)*RADIANCE(1:N,IMU,IXG(I1)) &
                             + SRCINT(2,IMU,I1)*DOSOURCE(1:N,IMU,IXG(I1)) &
                             + SRCINT(3,IMU,I1)*DOSOURCE(1:N,IMU,IXG(I1+1)) &
                             + SRCINT(4,IMU,I1)*DOSOURCE(1:N,IMU,IXG(I1+2))
        DO I = I1+1, I2-1
          RADIANCE(1:N,IMU,IXG(I+1)) = SRCINT(0,IMU,I)*RADIANCE(1:N,IMU,IXG(I)) &
                             + SRCINT(1,IMU,I)*DOSOURCE(1:N,IMU,IXG(I-1)) &
                             + SRCINT(2,IMU,I)*DOSOURCE(1:N,IMU,IXG(I)) &
                             + SRCINT(3,IMU,I)*DOSOURCE(1:N,IMU,IXG(I+1)) &
                             + SRCINT(4,IMU,I)*DOSOURCE(1:N,IMU,IXG(I+2))
        ENDDO
        RADIANCE(1:N,IMU,IXG(I2+1)) = SRCINT(0,IMU,I2)*RADIANCE(1:N,IMU,IXG(I2)) &
                             + SRCINT(1,IMU,I2)*DOSOURCE(1:N,IMU,IXG(I2-1)) &
                             + SRCINT(2,IMU,I2)*DOSOURCE(1:N,IMU,IXG(I2)) &
                             + SRCINT(3,IMU,I2)*DOSOURCE(1:N,IMU,IXG(I2+1))
      ENDIF
      IF (LAY<NLAY) RADIANCE(:,IMU,IXG(IXP(1,LAY+1))) = RADIANCE(:,IMU,IXG(IXP(2,LAY)))
    ENDDO

     ! Integrate over the DO radiances to get the downwelling surface flux
    DO K = 1, NPHI0(IMU)
      FLUXDNSFC = FLUXDNSFC &
                + ABS(MU(IMU))*WTDO(IMU,K)*RADIANCE(K,IMU,IXG(IXP(2,NLAY)))
    ENDDO
  ENDDO
 
    ! Now do all the surface reflection stuff
  IF (SRCTYPE /= 'T') THEN
    DIRFLUXSFC = SOLARFLUX*EXP(-TAUGSC(IXP(2,NLAY))/ABS(SOLARMU))
  ENDIF
  SFCPLANCK = 0.0
  IF (SRCTYPE /= 'S') THEN
    SFCPLANCK = PLANCKFUNC(WAVELEN,WAVENO,SFCTEMP)
  ENDIF

  ! For a Lambertian surface calculate the isotropic surface radiance
  !   from the thermal emission or direct reflection and the reflected 
  !   downwelling flux at the surface.
  IF (SFCTYPE == 'L') THEN
    ALB = SFCPARMS(1)
    IF (SRCTYPE == 'T') THEN
      BOTRAD = (1.0-ALB)*SFCPLANCK
    ELSE IF (SRCTYPE == 'S') THEN
      BOTRAD = ALB*DIRFLUXSFC/PI
    ELSE IF (SRCTYPE .EQ. 'B') THEN
      BOTRAD = (1.0-ALB)*SFCPLANCK + ALB*DIRFLUXSFC/PI
    ENDIF
    BOTRAD = BOTRAD + FLUXDNSFC*ALB/PI
    RADIANCE(:,NMU/2+1:NMU,IXG(IXP(2,NLAY))) = BOTRAD
  ELSE
    ! If not a Lambertian surface, compute the radiance for each of the
    !   upwelling ordinates by integrating the BRDF over the stored 
    !   downwelling radiances.
    DO IMU = NMU/2+1, NMU
      DO IPHI = 1, NPHI0(IMU)
        CALL VARIABLE_BRDF_SURFACE (NMU, NPHI0MAX, NPHI0, MU, PHI, WTDO, &
                                    MU(IMU), PHI(IPHI,IMU), &
                                    SRCTYPE, SOLARMU, DIRFLUXSFC, &
                                    SFCTYPE, SFCPARMS, SFCPLANCK, &
                                    RADIANCE(:,:,IXG(IXP(2,NLAY))), BOTRAD)
        RADIANCE(IPHI,IMU,IXG(IXP(2,NLAY))) = BOTRAD
      ENDDO
    ENDDO
  ENDIF


  ! Then integrate the upwelling zenith angles
      ! For upwelling ordinates integrate source function from the bottom up
  DO IMU = NMU/2+1, NMU
    N = NPHI0(IMU)
    DO LAY = NLAY, 1, -1
      I1 = IXP(1,LAY)
      I2 = IXP(2,LAY)-1
      IF (I1 == I2) THEN
        RADIANCE(1:N,IMU,IXG(I1)) = SRCINT(0,IMU,I1)*RADIANCE(1:N,IMU,IXG(I1+1)) &
                             + SRCINT(1,IMU,I1)*DOSOURCE(1:N,IMU,IXG(I1)) &
                             + SRCINT(2,IMU,I1)*DOSOURCE(1:N,IMU,IXG(I1+1))
      ELSE
        RADIANCE(1:N,IMU,IXG(I2)) = SRCINT(0,IMU,I2)*RADIANCE(1:N,IMU,IXG(I2+1)) &
                             + SRCINT(1,IMU,I2)*DOSOURCE(1:N,IMU,IXG(I2-1)) &
                             + SRCINT(2,IMU,I2)*DOSOURCE(1:N,IMU,IXG(I2)) &
                             + SRCINT(3,IMU,I2)*DOSOURCE(1:N,IMU,IXG(I2+1))
        DO I = I2-1, I1+1, -1
          RADIANCE(1:N,IMU,IXG(I)) = SRCINT(0,IMU,I)*RADIANCE(1:N,IMU,IXG(I+1)) &
                             + SRCINT(1,IMU,I)*DOSOURCE(1:N,IMU,IXG(I-1)) &
                             + SRCINT(2,IMU,I)*DOSOURCE(1:N,IMU,IXG(I)) &
                             + SRCINT(3,IMU,I)*DOSOURCE(1:N,IMU,IXG(I+1)) &
                             + SRCINT(4,IMU,I)*DOSOURCE(1:N,IMU,IXG(I+2))
        ENDDO
        RADIANCE(1:N,IMU,IXG(I1)) = SRCINT(0,IMU,I1)*RADIANCE(1:N,IMU,IXG(I1+1)) &
                             + SRCINT(2,IMU,I1)*DOSOURCE(1:N,IMU,IXG(I1)) &
                             + SRCINT(3,IMU,I1)*DOSOURCE(1:N,IMU,IXG(I1+1)) &
                             + SRCINT(4,IMU,I1)*DOSOURCE(1:N,IMU,IXG(I1+2))
      ENDIF
      IF (LAY>1) RADIANCE(:,IMU,IXG(IXP(2,LAY-1))) = RADIANCE(:,IMU,IXG(IXP(1,LAY)))
    ENDDO
  ENDDO

  LASTNPTS=NPTS
END SUBROUTINE PATH_INTEGRATION





SUBROUTINE SPLIT_CELLS (NLAY, NPTS, MAXIG, IXP, IXG, TAUGSC, &
                        NMU, NPHI0MAX, NPHI0, MU, DOSOURCE, RADIANCE, &
                        SPLITACC, TRUESOLCRIT, CURSPLITACC)
 ! Determines the current cell splitting accuracy (CURSPLITACC) from the
 ! solution accuracy (SOLCRIT) and the desired final cell splitting accuracy
 ! (SPLITACC).  Calculates the cell splitting criterion for each sublayer, 
 ! The cell splitting criterion for a sublayer is the rms difference over 
 ! outgoing ordinates in radiance between the original cubic source function
 ! interpolation integration (in RADIANCE) and a linear interpolation 
 ! integration, normalized by the rms of RADIANCE over all ordinates and 
 ! levels.  Splits cells with criterion greater than the current splitting
 ! accuracy.  Interpolates the discrete ordinate source function to the new
 ! levels.  The number of levels (NPTS), the adaptive index arrays (IXP, IXG),
 ! and the optical depth grid array (TAUGSC) are also updated.
 ! The boundary condition information is not needed for this routine because
 ! it has already been computed by PATH_INTEGRATION and stored in RADIANCE.
 ! SPLITCRITRATIO is the factor needed to divide the cell splitting criterion
 ! to convert it to a typical radiance accuracy of the split sublayer with
 ! the cubic interpolation integration.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY, MAXIG
  INTEGER, INTENT(INOUT) :: NPTS, IXP(2,NLAY), IXG(*)
  INTEGER, INTENT(IN) :: NMU, NPHI0MAX, NPHI0(NMU)
  REAL,    INTENT(IN) :: MU(NMU)
  REAL,    INTENT(IN) :: SPLITACC, TRUESOLCRIT
  REAL,    INTENT(IN) :: RADIANCE(NPHI0MAX,NMU,NPTS)
  REAL,    INTENT(INOUT) :: CURSPLITACC
  REAL,    INTENT(INOUT) :: TAUGSC(*), DOSOURCE(NPHI0MAX,NMU,*)
  INTEGER :: LAY, I, IMU, IMU2, N
  INTEGER :: IXPN(2,NLAY), NEWNPTS, I1, I2, I3, I4
  REAL    :: SPLITCRIT, NORM, T1, T2, T3, T4
  DOUBLE PRECISION  :: TAU(NMU/2), TR(NMU/2)
  DOUBLE PRECISION, ALLOCATABLE :: SRCINT1(:,:,:), DIFFRAD(:,:)

  NORM = SQRT( SUM(RADIANCE(:,:,1:NPTS)**2)/(NPTS*SUM(NPHI0(:))) )
   ! Leave immediately if splitting is not desired
  IF (SPLITACC <= 0.0 .OR. NORM == 0.0) RETURN

  ALLOCATE (SRCINT1(0:2,NMU/2,NPTS), DIFFRAD(NPHI0MAX,NMU))
  DIFFRAD(:,:) = 0.0

  CURSPLITACC = MAX(SPLITACC,MIN(CURSPLITACC,TRUESOLCRIT))

  ! Calculate and store the linear source function integration coefficients
  DO LAY = 1, NLAY
    DO I = IXP(1,LAY), IXP(2,LAY)-1
      TAU(:) = (TAUGSC(I+1) - TAUGSC(I))/ABS(MU(1:NMU/2))
      TR(:) = EXP(-TAU(:))        ! sublayer transmission along ordinate
      SRCINT1(0,:,I) = TR(:)
      IF (TAU(1) < 1.0E-4) THEN
        SRCINT1(1,:,I) = 0.5*TAU(:) - (1.0D0/3)*TAU(:)**2
        SRCINT1(2,:,I) = 0.5*TAU(:) - (1.0D0/6)*TAU(:)**2
      ELSE
        SRCINT1(1,:,I) = (1-TR(:)*(1+TAU(:)))/TAU(:)
        SRCINT1(2,:,I) = (TAU(:)-(1-TR(:)))/TAU(:)
      ENDIF
    ENDDO
  ENDDO

  ! For each sublayer integrate the downwelling zenith angles from top to
  ! bottom and the upwelling angles from bottom to top using linear 
  ! source function interpolation and difference from the previous
  ! cubic/quadratic interpolation results.
  NEWNPTS = NPTS
  IXPN(:,:) = IXP(:,:)
  DO LAY = 1, NLAY
   IF (IXP(1,LAY) < IXP(2,LAY)-1) THEN
    DO I = IXP(1,LAY), IXP(2,LAY)-1
      DO IMU = 1, NMU/2
        IMU2 = IMU + NMU/2
        N = NPHI0(IMU)
        DIFFRAD(1:N,IMU) = -RADIANCE(1:N,IMU,IXG(I+1)) &
                           + SRCINT1(0,IMU,I)*RADIANCE(1:N,IMU,IXG(I)) &
                           + SRCINT1(1,IMU,I)*DOSOURCE(1:N,IMU,IXG(I)) &
                           + SRCINT1(2,IMU,I)*DOSOURCE(1:N,IMU,IXG(I+1))
        DIFFRAD(1:N,IMU2) = -RADIANCE(1:N,IMU2,IXG(I)) &
                           + SRCINT1(0,IMU,I)*RADIANCE(1:N,IMU2,IXG(I+1)) &
                           + SRCINT1(1,IMU,I)*DOSOURCE(1:N,IMU2,IXG(I+1)) &
                           + SRCINT1(2,IMU,I)*DOSOURCE(1:N,IMU2,IXG(I))
      ENDDO
      SPLITCRIT = SQRT( SUM(DIFFRAD(:,:)**2)/SUM(NPHI0(:)) ) /NORM
        ! If the splitting criterion is large enough then add a level in
        !   the center of the current sublayer.  Put the new level at the
        !   end of the radiance array and update the pointers.
      IF (SPLITCRIT>CURSPLITACC) THEN
        IF (NEWNPTS>=MAXIG) THEN
          WRITE (*,*) 
          WRITE (*,*) 'Adaptive grid exceeded MAXIG=',MAXIG
          WRITE (*,*) 'Increase SPLITTING_FACTOR'
          STOP
        ENDIF
        NEWNPTS = NEWNPTS + 1
        TAUGSC(NEWNPTS) = 0.5*(TAUGSC(I)+TAUGSC(I+1))
        ! print *, newnpts,taug(newnpts), splitcrit
        IXG(NEWNPTS) = NEWNPTS
        IXPN(2,LAY:NLAY) = IXPN(2,LAY:NLAY) + 1
        IXPN(1,LAY+1:NLAY) = IXPN(1,LAY+1:NLAY) + 1
        I1 = MAX(I-1,IXP(1,LAY))  ;   I2 = I
        I3 = I+1  ; I4 = MIN(I+2,IXP(2,LAY))
        CALL INTERP_SOURCE (NPHI0MAX, NMU, TAUGSC(I1), DOSOURCE(:,:,IXG(I1)), &
                            TAUGSC(I2), DOSOURCE(:,:,IXG(I2)), &
                            TAUGSC(I3), DOSOURCE(:,:,IXG(I3)), &
                            TAUGSC(I4), DOSOURCE(:,:,IXG(I4)), &
                            TAUGSC(NEWNPTS), DOSOURCE(:,:,IXG(NEWNPTS)) )
      ENDIF
    ENDDO
   ENDIF
  ENDDO

  IF (NEWNPTS > NPTS) THEN
    CALL SSORT (TAUGSC, IXG, NEWNPTS, 2)
    NPTS = NEWNPTS
    IXP(:,:) = IXPN(:,:)
     ! Make sure that the IXG pointers are in the correct order
    DO I = 1, NPTS-1
      IF (TAUGSC(I) == TAUGSC(I+1) .AND. IXG(I) > IXG(I+1)) THEN
        I1 = IXG(I)
        IXG(I) = IXG(I+1)
        IXG(I+1) = I1
      ENDIF
    ENDDO
  ENDIF
  DEALLOCATE (SRCINT1, DIFFRAD)
END SUBROUTINE SPLIT_CELLS
 


SUBROUTINE INTERP_SOURCE (NPH, NMU, T1,S1, T2,S2, T3,S3, T4,S4, T,S)
 ! Does either quadratic or cubic polynomial interpolation of the discrete
 ! ordinate source function as a function of optical depth.  The source 
 ! function (S1,S2,S3,S4) is specified at three or four optical depths 
 ! (T1,T2,T3,T4), interpolated to T, and output in S.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NPH, NMU
  REAL,    INTENT(IN) :: T1, T2, T3, T4, T
  REAL,    INTENT(IN) :: S1(NPH,NMU), S2(NPH,NMU), S3(NPH,NMU), S4(NPH,NMU)
  REAL,    INTENT(OUT) :: S(NPH,NMU)

  IF (ABS(T1-T2) < 4*SPACING(T1)) THEN
    S = (T-T3)*(T-T4)*S2/((T2-T3)*(T2-T4)) &
      + (T-T2)*(T-T4)*S3/((T3-T2)*(T3-T4)) &
      + (T-T2)*(T-T3)*S4/((T4-T2)*(T4-T3))
  ELSE IF (ABS(T3-T4) < 4*SPACING(T3)) THEN
    S = (T-T2)*(T-T3)*S1/((T1-T2)*(T1-T3)) &
      + (T-T1)*(T-T3)*S2/((T2-T1)*(T2-T3)) &
      + (T-T1)*(T-T2)*S3/((T3-T1)*(T3-T2))
  ELSE
    S = (T-T2)*(T-T3)*(T-T4)*S1/((T1-T2)*(T1-T3)*(T1-T4)) &
      + (T-T1)*(T-T3)*(T-T4)*S2/((T2-T1)*(T2-T3)*(T2-T4)) &
      + (T-T1)*(T-T2)*(T-T4)*S3/((T3-T1)*(T3-T2)*(T3-T4)) &
      + (T-T1)*(T-T2)*(T-T3)*S4/((T4-T1)*(T4-T2)*(T4-T3))
  ENDIF
END SUBROUTINE INTERP_SOURCE



SUBROUTINE VARIABLE_BRDF_SURFACE (NMU, NPHI0MAX, NPHI0, MU, PHI, WTDO, &
                                  MU2, PHI2, SRCTYPE, SOLARMU, DIRFLUXSFC, &
                                  SFCTYPE, SFCPARMS, SFCPLANCK, &
                                  SFCRADIANCE, BOTRAD)
 ! Computes the upwelling radiance at the bottom boundary for one 
 ! outgoing direction using the specified bidirectional reflectance 
 ! distribution function.  The upwelling radiance includes the reflection of
 ! the incident radiance, the thermal emission (emissivity*Planck function),
 ! and the reflected direct solar flux (if applicable).  The upwelling 
 ! radiance is the integral over all incident directions of the BRDF times 
 ! the downwelling radiance, so a discrete sum is done and the integral 
 ! weights (WTDO) are included.  The general BRDF function is called 
 ! to compute the reflectance for the particular BRDF type (SFCTYPE) 
 ! with parameters (SFCPARMS) for the incident and outgoing directions.  
 ! The emissivity is computed implicitly from the integral of the BRDF.  
 ! The outgoing direction is specified with (MU2,PHI2), and the BRDF is 
 ! computed for all incident directions (loop over JMU,JPHI).  The incident 
 ! downwelling radiances are input in SFCRADIANCE(:,:) and the outgoing 
 ! upwelling radiance is output in BOTRAD.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NMU, NPHI0MAX, NPHI0(NMU)
  REAL,    INTENT(IN) :: MU(NMU), PHI(NPHI0MAX,NMU), WTDO(NMU,NPHI0MAX)
  REAL,    INTENT(IN) :: MU2, PHI2, SOLARMU, DIRFLUXSFC, SFCPLANCK
  REAL,    INTENT(IN) :: SFCPARMS(*), SFCRADIANCE(NPHI0MAX,NMU)
  REAL,    INTENT(OUT):: BOTRAD
  CHARACTER(LEN=1), INTENT(IN) :: SRCTYPE, SFCTYPE
  INTEGER :: JMU, JPHI
  REAL    :: OPI, SOLARAZ, REFLECT, W

  OPI = 1.0/ACOS(-1.0)
  SOLARAZ = 0.0

  ! Initialize the upwelling boundary radiances to zero (for thermal)
  !   or to the reflected direct solar flux.
  IF (SRCTYPE == 'T') THEN
    BOTRAD = 0.0
  ELSE
    CALL SURFACE_BRDF (SFCTYPE, SFCPARMS, MU2, PHI2, SOLARMU, SOLARAZ, &
                       REFLECT)
    BOTRAD = OPI*REFLECT*DIRFLUXSFC
  ENDIF

  ! Integrate over the incident discrete ordinate directions (JMU,JPHI)
  DO JMU = 1, NMU/2
    DO JPHI = 1, NPHI0(JMU)
      CALL SURFACE_BRDF (SFCTYPE, SFCPARMS, MU2, PHI2, &
                         MU(JMU),PHI(JPHI,JMU), REFLECT)
      W = OPI*ABS(MU(JMU))*WTDO(JMU,JPHI)
      BOTRAD = BOTRAD + W*REFLECT*SFCRADIANCE(JPHI,JMU) &
                      + W*(1-REFLECT)*SFCPLANCK
    ENDDO
  ENDDO

END SUBROUTINE VARIABLE_BRDF_SURFACE




SUBROUTINE SURFACE_BRDF (SFCTYPE, REFPARMS, &
                         MU2, PHI2, MU1, PHI1, REFLECT)
 !  Returns the reflection coefficient for the general bidirectional
 ! reflection distribution function of the specified type (SFCTYPE).
 ! The incident direction is (MU1,PHI1), and the outgoing direction
 ! is (MU2,PHI2) (MU is cosine of zenith angle, and PHI is the azimuthal
 ! angle in radians).  The incident directions have mu<0 (downward), 
 ! while the outgoing directions have mu>0 (upward). The reflection 
 ! function is normalized so that for a Lambertian surface (uniform 
 ! reflection) the returned value (REFLECT) is simply the albedo.
 ! This routine calls the desired BRDF function, passing the 
 ! appropriate parameters.  More BRDF surface types may be added easily 
 ! by putting in the appropriate function calls.
 !        Type        Parameters
 !    R  Rahman et al  rho0, k, Theta
  IMPLICIT NONE
  REAL, INTENT(IN) :: REFPARMS(*), MU1, PHI1, MU2, PHI2
  REAL, INTENT(OUT):: REFLECT
  CHARACTER(LEN=1), INTENT(IN) :: SFCTYPE
  REAL  :: PI
  REAL  :: RPV_REFLECTION

  PI = ACOS(-1.0)
  IF (SFCTYPE == 'R') THEN
    ! R: Rahman, Pinty, and Verstraete
    REFLECT = RPV_REFLECTION (REFPARMS(1),REFPARMS(2),REFPARMS(3), &
                              -MU1, MU2, PHI1-PHI2-PI)
  ELSE
    STOP 'SURFACE_BRDF: Unknown BRDF type'
  ENDIF
END SUBROUTINE SURFACE_BRDF




FUNCTION RPV_REFLECTION (RHO0, K, THETA, MU1, MU2, PHI)
 !  Computes the Rahman, Pinty, Verstraete BRDF.  The incident
 ! and outgoing cosine zenith angles are MU1 and MU2, respectively,
 ! and the relative azimuthal angle is PHI.  In this case the incident
 ! direction is where the radiation is coming from (i.e. opposite of the
 ! discrete ordinate), so MU1>0 and the hot spot is MU2=MU1 and PHI=0.
 ! The reference is:
 !    Rahman, Pinty, Verstraete, 1993: Coupled Surface-Atmosphere 
 !    Reflectance (CSAR) Model. 2. Semiempirical Surface Model Usable 
 !    With NOAA Advanced Very High Resolution Radiometer Data,
 !    J. Geophys. Res., 98, 20791-20801.
  IMPLICIT NONE
  REAL, INTENT(IN) :: RHO0, K, THETA, MU1, MU2, PHI
  REAL :: RPV_REFLECTION
  REAL M, F, H, COSPHI, SIN1, SIN2, COSG, TAN1, TAN2, CAPG

  M = MU1**(K-1) * MU2**(K-1) / (MU1 + MU2)**(1-K)
  COSPHI = COS(PHI)
  SIN1 = SQRT(1.0-MU1**2)
  SIN2 = SQRT(1.0-MU2**2)
  COSG = MU1*MU2 + SIN1*SIN2*COSPHI
  F = (1-THETA**2) / (1 + 2*THETA*COSG + THETA**2)**1.5
  TAN1 = SIN1/MU1
  TAN2 = SIN2/MU2
  CAPG = SQRT( TAN1**2 + TAN2**2 - 2*TAN1*TAN2*COSPHI )
  H = 1 + (1-RHO0)/(1+CAPG)
  RPV_REFLECTION = RHO0 * M * F * H
END FUNCTION RPV_REFLECTION





SUBROUTINE COMPUTE_ONE_SOURCE (NLAY, ALBEDOSC, LEGENSC, SSPHASE, DELTA_F, &
                               NPTS, IXP, ML, MM, NLM, &
                               MUO, PHIO, SRCTYPE, DELTAM, &
                               SOLARMU, DIRFLUX, SOURCE, SOURCE1)
 ! Computes the source function (SOURCE1) in the direction (MUO,PHIO)
 ! for all levels (NPTS).  The spherical harmonic source function
 ! series is input in SOURCE.  DIRFLUX is the attenuated direct beam 
 ! perpendicular flux.
 ! For a solar source if delta-M then use Nakajima and Tanaka TMS
 ! procedure, replacing delta-M single scattering with single scattering
 ! for unscaled untruncated phase function.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY, NPTS
  INTEGER, INTENT(IN) :: ML, MM, NLM, IXP(2,NLAY)
  LOGICAL, INTENT(IN) :: DELTAM
  CHARACTER(LEN=1), INTENT(IN) :: SRCTYPE
  REAL,    INTENT(IN) :: ALBEDOSC(NLAY), LEGENSC(1:ML,NLAY)
  REAL,    INTENT(IN) :: SSPHASE(NLAY), DELTA_F(NLAY)
  REAL,    INTENT(IN) :: MUO, PHIO, SOLARMU, DIRFLUX(NPTS)
  REAL,    INTENT(IN) :: SOURCE(NLM,NPTS)
  REAL,    INTENT(OUT) :: SOURCE1(NPTS)
  INTEGER :: I, J, K, L, LAY, M, ME
  DOUBLE PRECISION  :: A
  REAL, ALLOCATABLE :: YLMDIR(:), YLMSUN(:)

  ALLOCATE (YLMDIR(NLM), YLMSUN(NLM))

  ! Precompute Ylm's for output direction
  CALL YLMALL (MUO, PHIO, ML, MM, 1, YLMDIR)

  ! Special case for solar source and Delta-M
  IF (SRCTYPE /= 'T' .AND. DELTAM) THEN
    CALL YLMALL (SOLARMU, 0.0, ML, MM, 1, YLMSUN)

    ! Loop over all the grid points, computing the source function 
    !   at the viewing angle from the spherical harmonic source function.
    DO LAY = 1, NLAY
      DO K = IXP(1,LAY), IXP(2,LAY)
         ! Sum over the spherical harmonic series of the source function
        SOURCE1(K) = SUM(SOURCE(:,K)*YLMDIR(:))
         ! First subtract off the truncated single scattering 
        SOURCE1(K) = SOURCE1(K) - DIRFLUX(K)*ALBEDOSC(LAY)*YLMDIR(1)*YLMSUN(1)
        J = 2
        DO L = 1, ML
          ME = MIN(L,MM)
          A = DIRFLUX(K)*ALBEDOSC(LAY)*LEGENSC(L,LAY)/(2*L+1)
          DO M = 0, ME
            SOURCE1(K) = SOURCE1(K) - A*YLMDIR(J)*YLMSUN(J)
            J = J + 1
          ENDDO
        ENDDO
        ! Then add in the single scattering contribution.  
        !   Need to un-delta-scale the scattering optical depth
        SOURCE1(K) = SOURCE1(K) &
                   + DIRFLUX(K)* ALBEDOSC(LAY)*SSPHASE(LAY)/(1-DELTA_F(LAY))
      ENDDO
    ENDDO

  ELSE  ! for thermal sources or no delta-M it is very simple:
    ! Sum over the spherical harmonic series of the source function
    DO K = 1, NPTS
      SOURCE1(K) = SUM(SOURCE(:,K)*YLMDIR(:))
    ENDDO
  ENDIF

  DEALLOCATE (YLMDIR, YLMSUN)
END SUBROUTINE COMPUTE_ONE_SOURCE
 



SUBROUTINE INTEGRATE_SOURCE (NLAY, NPTS, IXP, DTAUGSC, &
                             MUO, BNDRAD, SOURCE1, RADVEC)
 ! Integrates the radiative transfer equation through the medium specified 
 ! by the sublayer delta optical depth (DTAUGSC) and source function (SOURCE1)
 ! in direction cosine MUO.  The radiance at the approprate boundary (top
 ! or bottom depending on the sign of MUO) is input (BNDRAD).  The radiance 
 ! for the direction of the source function at all the adaptive grid 
 ! levels is output (RADVEC with indexing corresponding to TAUGSC).
 ! Uses the same quadratic/cubic polynomial source function interpolation
 ! as in PATH_INTEGRATION.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY, NPTS, IXP(2,NLAY)
  REAL,    INTENT(IN) :: MUO, BNDRAD
  REAL,    INTENT(IN) :: DTAUGSC(NPTS), SOURCE1(NPTS)
  REAL,    INTENT(OUT) :: RADVEC(NPTS)
  INTEGER :: LAY, I, I1, I2
  REAL, ALLOCATABLE :: SRCINT(:,:)
  DOUBLE PRECISION :: A(4), B(4), C(4), D(4), E, T, T1, T3, T4, TR

  ! Calculate and store the source function integration coefficients
  ALLOCATE (SRCINT(0:4,NPTS))
  DO LAY = 1, NLAY
    I1 = IXP(1,LAY)
    I2 = IXP(2,LAY)-1
    IF (I1 == I2) THEN   ! special case for layer with just two points
      T = DTAUGSC(I1)/ABS(MUO)
      TR = EXP(-T)        ! sublayer transmission along ordinate
      IF (T < 1.0E-4) THEN
        D(1) = 0.5*T - (1.0D0/3)*T**2
        D(2) = 0.5*T - (1.0D0/6)*T**2
      ELSE
        D(1) = (1-TR*(1+T))/T
        D(2) = (T-(1-TR))/T
      ENDIF
      SRCINT(0,I1) = TR
      IF (MUO < 0.0) THEN
        SRCINT(1,I1) = D(2)
        SRCINT(2,I1) = D(1)
      ELSE
        SRCINT(1,I1) = D(1)
        SRCINT(2,I1) = D(2)
      ENDIF
    ELSE
      DO I = I1, I2
        A(:)=0.0 ; B(:)=0.0 ; C(:)=0.0 ; D(:)=0.0
        T3 = DTAUGSC(I)/ABS(MUO)
        IF (I == I1) THEN
          T4 = (DTAUGSC(I)+DTAUGSC(I+1))/ABS(MUO)
          E=T3*T4      ; B(2)=1/E ; C(2)=-(T3+T4)/E ; D(2)=1.0
          E=T3*(T3-T4) ; B(3)=1/E ; C(3)=-T4/E      ; D(3)=0.0
          E=T4*(T4-T3) ; B(4)=1/E ; C(4)=-T3/E      ; D(4)=0.0
        ELSE IF (I < I2) THEN
          T1 = -DTAUGSC(I-1)/ABS(MUO)
          T4 = (DTAUGSC(I)+DTAUGSC(I+1))/ABS(MUO)
          E=T1*(T1-T3)*(T1-T4) ; A(1)=1/E
          B(1)=-(T3+T4)/E    ; C(1)=T3*T4/E ; D(1)=0.0
          E=-T1*T3*T4 ; A(2)=1/E 
          B(2)=-(T1+T3+T4)/E ; C(2)=(T1*T3+T1*T4+T3*T4)/E ; D(2)=1.0
          E=T3*(T3-T1)*(T3-T4) ; A(3)=1/E
          B(3)=-(T1+T4)/E    ; C(3)=T1*T4/E ; D(3)=0.0
          E=T4*(T4-T1)*(T4-T3) ; A(4)=1/E
          B(4)=-(T1+T3)/E    ; C(4)=T1*T3/E ; D(4)=0.0
        ELSE
          T1 = -DTAUGSC(I-1)/ABS(MUO)
          E=T1*(T1-T3) ; B(1)=1/E ; C(1)=-T3/E      ; D(1)=0.0
          E=T1*T3      ; B(2)=1/E ; C(2)=-(T1+T3)/E ; D(2)=1.0
          E=T3*(T3-T1) ; B(3)=1/E ; C(3)=-T1/E      ; D(3)=0.0
        ENDIF
        T = T3
        TR = EXP(-T)
        SRCINT(0,I) = TR
        IF (MUO < 0.0) THEN
          SRCINT(1:4,I) = A(:)*(6*TR+T**3-3*T**2+6*T-6) + D(:)*(1-TR) &
                        + B(:)*(-2*TR+T**2-2*T+2) + C(:)*(TR+T-1)
        ELSE        
          SRCINT(1:4,I) = A(:)*(6-TR*(T**3+3*T**2+6*T+6)) + D(:)*(1-TR) &
                        + B(:)*(2-TR*(T**2+2*T+2)) + C(:)*(1-TR*(T+1)) 
        ENDIF
      ENDDO
    ENDIF
  ENDDO

!$TAF INIT tape_integrate_source = MEMORY

  IF (MUO < 0.0) THEN
    ! For downwelling direction integrate source function from the top down
!$TAF STORE RADVEC = tape_integrate_source
    RADVEC(IXP(1,1)) = BNDRAD
    DO LAY = 1, NLAY
      I1 = IXP(1,LAY)
      I2 = IXP(2,LAY)-1
      IF (I1 == I2) THEN
!$TAF STORE RADVEC = tape_integrate_source
        RADVEC(I1+1) = SRCINT(0,I1)*RADVEC(I1) &
             + SRCINT(1,I1)*SOURCE1(I1) + SRCINT(2,I1)*SOURCE1(I1+1)
      ELSE
!$TAF STORE RADVEC = tape_integrate_source
        RADVEC(I1+1) = SRCINT(0,I1)*RADVEC(I1) + SRCINT(2,I1)*SOURCE1(I1) &
             + SRCINT(3,I1)*SOURCE1(I1+1) + SRCINT(4,I1)*SOURCE1(I1+2)
        DO I = I1+1, I2-1
!$TAF STORE RADVEC = tape_integrate_source
          RADVEC(I+1) = SRCINT(0,I)*RADVEC(I) + SRCINT(1,I)*SOURCE1(I-1) &
               + SRCINT(2,I)*SOURCE1(I) + SRCINT(3,I)*SOURCE1(I+1) &
               + SRCINT(4,I)*SOURCE1(I+2)
        ENDDO
!$TAF STORE RADVEC = tape_integrate_source
        RADVEC(I2+1) = SRCINT(0,I2)*RADVEC(I2) + SRCINT(1,I2)*SOURCE1(I2-1) &
             + SRCINT(2,I2)*SOURCE1(I2) + SRCINT(3,I2)*SOURCE1(I2+1)
      ENDIF
!$TAF STORE RADVEC = tape_integrate_source
      IF (LAY<NLAY) RADVEC(IXP(1,LAY+1)) = RADVEC(IXP(2,LAY))
    ENDDO
  ELSE
    ! For upwelling direction integrate source function from the bottom up
!$TAF STORE RADVEC = tape_integrate_source
    RADVEC(IXP(2,NLAY)) = BNDRAD
    DO LAY = NLAY, 1, -1
      I1 = IXP(1,LAY)
      I2 = IXP(2,LAY)-1
      IF (I1 == I2) THEN
!$TAF STORE RADVEC = tape_integrate_source
        RADVEC(I1) = SRCINT(0,I1)*RADVEC(I1+1) &
             + SRCINT(1,I1)*SOURCE1(I1) + SRCINT(2,I1)*SOURCE1(I1+1)
      ELSE
!$TAF STORE RADVEC = tape_integrate_source
        RADVEC(I2) = SRCINT(0,I2)*RADVEC(I2+1) + SRCINT(1,I2)*SOURCE1(I2-1) &
             + SRCINT(2,I2)*SOURCE1(I2) + SRCINT(3,I2)*SOURCE1(I2+1)
        DO I = I2-1, I1+1, -1
!$TAF STORE RADVEC = tape_integrate_source
          RADVEC(I) = SRCINT(0,I)*RADVEC(I+1) + SRCINT(1,I)*SOURCE1(I-1) &
             + SRCINT(2,I)*SOURCE1(I) + SRCINT(3,I)*SOURCE1(I+1) &
             + SRCINT(4,I)*SOURCE1(I+2)
        ENDDO
!$TAF STORE RADVEC = tape_integrate_source
        RADVEC(I1) = SRCINT(0,I1)*RADVEC(I1+1) + SRCINT(2,I1)*SOURCE1(I1) &
            + SRCINT(3,I1)*SOURCE1(I1+1) + SRCINT(4,I1)*SOURCE1(I1+2)
      ENDIF
!$TAF STORE RADVEC = tape_integrate_source
      IF (LAY>1) RADVEC(IXP(2,LAY-1)) = RADVEC(IXP(1,LAY))
    ENDDO
  ENDIF
  DEALLOCATE (SRCINT)
END SUBROUTINE INTEGRATE_SOURCE
 



SUBROUTINE INIT_RADIANCE (NLAY, ML, TAUPSC, ALBEDOSC, LEGENSC, TEMPP, &
                          SRCTYPE, SOLARFLUX, SOLARMU, GNDALB, SFCTEMP, &
                          SKYRAD, WAVELEN, WAVENO,  NPTS, TAUGSC, IXP, NLM, &
                          SHRADIANCE)
 ! Initializes the spherical harmonic radiance field by solving the
 ! plane-parallel Eddington system to give the first two terms of the
 ! spherical harmonic series at the NPTS sublayer interfaces.
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAY, ML, NPTS, IXP(2,NLAY), NLM
  REAL,    INTENT(IN) :: TAUPSC(NLAY), ALBEDOSC(NLAY), LEGENSC(1:ML,NLAY)
  REAL,    INTENT(IN) :: TEMPP(NLAY+1)
  REAL,    INTENT(IN) :: SOLARFLUX, SOLARMU, GNDALB, SFCTEMP
  REAL,    INTENT(IN) :: SKYRAD, WAVELEN, WAVENO(2), TAUGSC(NPTS)
  CHARACTER(LEN=1), INTENT(IN) :: SRCTYPE
  REAL,    INTENT(OUT):: SHRADIANCE(NLM,NPTS)

  INTEGER :: NLAYER, LAY, I, L
  LOGICAL, PARAMETER :: DELTAM = .FALSE.
  REAL    :: PI, C0, C1, GNDEMIS, TAU1, U
  REAL, ALLOCATABLE :: OPTDEPTHS(:), ALBEDOS(:), ASYMMETRIES(:)
  REAL, ALLOCATABLE :: TEMPS(:), FLUXES(:,:)
  CHARACTER(LEN=1) :: SRCTYPE2

  NLAYER = SUM(IXP(2,:)-IXP(1,:))
  ALLOCATE (OPTDEPTHS(NLAYER), ALBEDOS(NLAYER), ASYMMETRIES(NLAYER))
  ALLOCATE (TEMPS(NLAYER+1), FLUXES(3,NLAYER+1))

  SRCTYPE2 = SRCTYPE
  IF (SRCTYPE == 'B') SRCTYPE2='T'
  PI = ACOS(-1.0)
  C0 = SQRT(1.0/PI)
  C1 = SQRT(3.0/(4*PI))

  ! Make layer properties for the Eddington routine
  L = 0
  DO LAY = 1, NLAY
    TAU1 = TAUGSC(IXP(1,LAY))
    DO I = IXP(1,LAY), IXP(2,LAY)-1
      L = L + 1
      OPTDEPTHS(L) = TAUGSC(I+1)-TAUGSC(I)
      ALBEDOS(L) = ALBEDOSC(LAY)
      ASYMMETRIES(L) = LEGENSC(1,LAY)/3.0
      IF (TAUPSC(LAY) > 0.0) THEN
        U = (TAUGSC(I)-TAU1)/TAUPSC(LAY)
      ELSE
        U = (I-IXP(1,LAY))/FLOAT(IXP(2,LAY)-IXP(1,LAY))
      ENDIF
      TEMPS(L) = TEMPP(LAY) + U*(TEMPP(LAY+1)-TEMPP(LAY))
    ENDDO
  ENDDO
  IF (L /= NLAYER) STOP 'INIT_RADIANCE: initial grid error'
  TEMPS(NLAYER+1) = TEMPP(NLAY+1)

     ! Call the Eddington flux routine
  GNDEMIS = 1.0-GNDALB
  CALL EDDRTF (NLAYER, OPTDEPTHS, ALBEDOS, ASYMMETRIES, &
               TEMPS, DELTAM, SRCTYPE2, SOLARFLUX, SOLARMU, &
               SFCTEMP, GNDEMIS, SKYRAD, WAVELEN, WAVENO, FLUXES)


  ! Convert fluxes to first two moments of spherical harmonics
  SHRADIANCE(:,:) = 0.0
  L = 0
  DO LAY = 1, NLAY
    DO I = IXP(1,LAY), IXP(2,LAY)
      L = L + 1
      SHRADIANCE(1,I) = C0*(FLUXES(1,L)+FLUXES(2,L))
      SHRADIANCE(2,I) = C1*(FLUXES(1,L)-FLUXES(2,L))
    ENDDO
    L = L - 1
  ENDDO

  DEALLOCATE (OPTDEPTHS, ALBEDOS, ASYMMETRIES, TEMPS, FLUXES)
END SUBROUTINE INIT_RADIANCE


 
 

SUBROUTINE EDDRTF (NLAYER, OPTDEPTHS, ALBEDOS, ASYMMETRIES, &
                   TEMPS, DELTAM, SRCTYPE, SOLARFLUX, SOLARMU, &
                   SFCTEMP, GNDEMIS, SKYRAD, WAVELEN, WAVENO,  FLUXES)
 ! EDDRTF computes the layer interface fluxes for a plane-parallel
 ! atmosphere with either solar or thermal source of radiation using 
 ! the Eddington approximation.  The medium is specified by a number 
 ! of homogeneous layers.  For a thermal source the Planck function is
 ! linear with optical depth, while for a solar source it is exponential.
 ! The temperatures, optical depth, single scattering albedo, and 
 ! asymmetry parameter are specified for each layer.  The boundary
 ! conditions such as the solar flux, and reflection and/or emission from 
 ! ground surface are also specified. Delta Eddington scaling may be 
 ! used.  The diffuse Eddington fluxes and the solar direct flux at 
 ! each level are returned.
 ! The model works by calculating the reflection, transmission, and
 ! source terms for each layer from the input properties.  A
 ! tri-diagonal matrix solver is then used to compute the diffuse fluxes 
 ! at each layer interface from the applied boundary conditions.
 !
 ! Parameters:
 !   Input:
 ! NLAYER         integer      Number of homogenous layers
 !                              (layers are specified from the top down)
 ! OPTDEPTHS      real array   Optical thickness of layers
 ! ALBEDOS        real array   Single scattering albedos
 ! ASYMMETRIES    real array   Asymmetry parameters
 ! TEMPS          real array   Temperatures (K) at layer interfaces
 !                              (e.g. TEMPS(1) is at top of top layer, 
 !                               TEMPS(2) is at bottom of top layer).
 ! DELTAM         logical      True for delta-Eddington scaling
 ! SRCTYPE        character    'S' for solar source, 'T' for thermal source
 ! SOLARFLUX      real         Incident solar flux on horizonal plane
 ! SOLARMU        real         Cosine of the solar zenith angle
 ! SFCTEMP        real         Surface temperature (Kelvin)
 ! GNDEMIS        real         Ground emissivity (1-albedo)
 ! SKYRAD         real         Radiance (for solar) or brightness 
 !                              temperature (for thermal) of isotropic 
 !                              incident radiation from above
 ! WAVELEN        real         Wavelength (micron)
 ! WAVENO(2)      real         Wavenumber range (cm^-1) (if WAVELEN=0)
 !
 !   Output:
 ! FLUXES         real         Eddington fluxes at layer interfaces.
 !                               FLUXES(1,L) is upwelling diffuse, 
 !                               FLUXES(2,L) is downwelling diffuse,
 !                               FLUXES(3,L) is downwelling direct,
 !                               L=1 is top, L=NLAYER+1 is bottom
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: NLAYER
  LOGICAL, INTENT(IN) :: DELTAM
  REAL,    INTENT(IN) :: TEMPS(NLAYER+1)
  REAL,    INTENT(IN) :: OPTDEPTHS(NLAYER), ALBEDOS(NLAYER), ASYMMETRIES(NLAYER)
  REAL,    INTENT(IN) :: SFCTEMP, GNDEMIS, SKYRAD, SOLARFLUX, SOLARMU
  REAL,    INTENT(IN) :: WAVELEN, WAVENO(2)
  CHARACTER(LEN=1), INTENT(IN) ::  SRCTYPE
  REAL,    INTENT(OUT) :: FLUXES(3,NLAYER+1)
 
  INTEGER :: N, L, I
  DOUBLE PRECISION :: DELTAU, G, OMEGA, F
  DOUBLE PRECISION :: LAMBDA, R, T, D, CP, CM, A, B, X1, X2
  DOUBLE PRECISION :: REFLECT, TRANS, SOURCEP, SOURCEM
  DOUBLE PRECISION :: RADP1P, RADP1M, RADP2P, RADP2M
  DOUBLE PRECISION :: PI, MU0, SKYFLUX,GNDFLUX, PLANCK1,PLANCK2,C,TAU
  DOUBLE PRECISION :: EXLP, EXLM, V, DS, B1, B2, SOLPP, SOLPM
  DOUBLE PRECISION, ALLOCATABLE :: LOWER(:), UPPER(:), DIAG(:), RHS(:)
  PARAMETER (PI=3.1415926535)
  REAL :: PLANCKFUNC

  ! Compute the reflection, transmission, and source coefficients 
  !   for each layer for the diffuse Eddington two stream problem.
  N = 2*NLAYER+2
  ALLOCATE (LOWER(N), UPPER(N), DIAG(N), RHS(N))
  IF (SRCTYPE == 'T') THEN
    PLANCK1 = PI*PLANCKFUNC(WAVELEN,WAVENO,TEMPS(1))
  ENDIF
  MU0 = ABS(SOLARMU)
  TAU = 0.0
  I = 2
  DO L = 1, NLAYER
    DELTAU = OPTDEPTHS(L)
    IF (DELTAU < 0.0) STOP 'EDDRTF: TAU<0'
    ! Special case for zero optical depth
    IF (DELTAU == 0.0) THEN
      TRANS = 1.0
      REFLECT = 0.0
      SOURCEP = 0.0
      SOURCEM = 0.0
    ELSE
      OMEGA = ALBEDOS(L)
      G = ASYMMETRIES(L)
      IF (DELTAM) THEN
        F = G**2
        DELTAU = (1-OMEGA*F)*DELTAU
        OMEGA = (1-F)*OMEGA/(1-OMEGA*F)
        G = (G-F)/(1-F)
      ENDIF
      R = ( 1.0 - OMEGA*(4.0-3.0*G) )/4.0
      T = ( 7.0 - OMEGA*(4.0+3.0*G) )/4.0
      LAMBDA = SQRT( 3.0*(1.0-OMEGA)*(1.0-OMEGA*G) )
      ! Special case for conservative scattering (lambda=0)
      IF (LAMBDA == 0.0) THEN
        D = 1.0/(1.0+T*DELTAU)
        TRANS = D
        REFLECT = -R*DELTAU*D
      ELSE
        X1 = -R
        X2 = LAMBDA + T
        EXLP = DEXP(MIN(LAMBDA*DELTAU,75.D0))
        EXLM = 1.0/EXLP
        TRANS = 2.*LAMBDA/(X2*EXLP + (LAMBDA-T)*EXLM)
        REFLECT = X1*(EXLP - EXLM) *TRANS /(2.*LAMBDA)
        D = 1.0/(X2**2 *EXLP - X1**2 *EXLM)
      ENDIF

      IF (SRCTYPE == 'T') THEN
        ! Calculate thermal source terms
        PLANCK2 = PI*PLANCKFUNC(WAVELEN,WAVENO,TEMPS(L+1))
        V = 2.0*(PLANCK2-PLANCK1)/(3.0*(1.-OMEGA*G)*DELTAU)
        RADP1P = -V + PLANCK1
        RADP2M =  V + PLANCK2
        RADP2P = -V + PLANCK2
        RADP1M =  V + PLANCK1
        IF (LAMBDA .EQ. 0.0) THEN
          A =  (R*DELTAU*RADP1P - RADP2M) *D
          B = -(R*RADP1P + T*RADP2M) *D
          SOURCEP = (B - T*(A+B*DELTAU))/R + RADP2P
          SOURCEM = A + RADP1M
        ELSE
          CP  =  (X1*EXLM*RADP1P - X2*RADP2M) *D
          CM = (-X2*EXLP*RADP1P + X1*RADP2M) *D
          SOURCEP = X1*CP*EXLP + X2*CM*EXLM + RADP2P
          SOURCEM = X2*CP + X1*CM + RADP1M
        ENDIF
        PLANCK1 = PLANCK2
        FLUXES(3,L) = 0.0
      ELSE
        ! Calculate solar source terms
        FLUXES(3,L) = SOLARFLUX*EXP(-TAU/MU0)
        DS = 1.0/(LAMBDA**2-1.0/MU0**2)
        B1 = 0.5*OMEGA*(SOLARFLUX/MU0)*EXP(-TAU/MU0) *DS
        B2 = 0.5*OMEGA*(SOLARFLUX/MU0)*EXP(-(TAU+DELTAU)/MU0) *DS
        SOLPP =  1.0 + 1.5*G*MU0
        SOLPM = -1.0 + 1.5*G*MU0
        RADP1P = ( (T+1.0/MU0)*SOLPP + R*SOLPM )*B1
        RADP2M = ((-T+1.0/MU0)*SOLPM - R*SOLPP )*B2
        RADP2P = ( (T+1.0/MU0)*SOLPP + R*SOLPM )*B2
        RADP1M = ((-T+1.0/MU0)*SOLPM - R*SOLPP )*B1
        IF (LAMBDA .EQ. 0.0) THEN
          A =  (R*DELTAU*RADP1P - RADP2M) *D
          B = -(R*RADP1P + T*RADP2M) *D
          SOURCEP = (B - T*(A+B*DELTAU))/R + RADP2P
          SOURCEM = A + RADP1M
        ELSE
          CP  =  (X1*EXLM*RADP1P - X2*RADP2M) *D
          CM = (-X2*EXLP*RADP1P + X1*RADP2M) *D
          SOURCEP = X1*CP*EXLP + X2*CM*EXLM + RADP2P
          SOURCEM = X2*CP + X1*CM + RADP1M
        ENDIF
        TAU = TAU + DELTAU
      ENDIF
    ENDIF
    DIAG(I) = -REFLECT
    DIAG(I+1) = -REFLECT
    LOWER(I) = 1.0
    LOWER(I+1) = -TRANS
    UPPER(I) = -TRANS
    UPPER(I+1) = 1.0
    RHS(I) = SOURCEM
    RHS(I+1) = SOURCEP
    I = I + 2
  ENDDO

  ! Set up boundary radiances
  IF (SRCTYPE == 'S') THEN
    FLUXES(3,NLAYER+1) = SOLARFLUX*EXP(-TAU/MU0)
    GNDFLUX = (1.0-GNDEMIS)*SOLARFLUX*EXP(-TAU/MU0)
    SKYFLUX = PI*SKYRAD
  ELSE
    FLUXES(3,NLAYER+1) = 0.0
    GNDFLUX = PI*GNDEMIS*PLANCKFUNC(WAVELEN,WAVENO,SFCTEMP)
    SKYFLUX = PI*PLANCKFUNC(WAVELEN,WAVENO,SKYRAD)
  ENDIF
  ! Setup for and call the tri-diagonal matrix solver
  RHS(1) = SKYFLUX
  DIAG(1) = 0.0
  UPPER(1) = 1.0
  DIAG(N) = -(1.0-GNDEMIS)
  LOWER(N) = 1.0
  RHS(N) = GNDFLUX
  CALL TRIDIAG (N, LOWER, DIAG, UPPER, RHS)
  ! Put the fluxes in the output array
  I = 1
  DO L = 1, NLAYER+1 
    FLUXES(1,L) = RHS(I)
    FLUXES(2,L) = RHS(I+1)
    I = I + 2
  ENDDO
 
  DEALLOCATE (LOWER, UPPER, DIAG, RHS)
END SUBROUTINE EDDRTF
 


SUBROUTINE TRIDIAG (N, LOWER, DIAG, UPPER, RHS)
 ! Computes the solution to a tridiagonal system. 
 ! N is order of the matrix.  LOWER(2..N) is the subdiagonal,
 ! DIAG(1..N) is the diagonal, and UPPER(1..N-1) is the 
 ! superdiagonal.  On input RHS is the right hand side, while
 ! on output it is the solution vector.  Everything is destroyed.
 ! Hacked from Linpack DGTSL.
  IMPLICIT NONE
  INTEGER :: N 
  DOUBLE PRECISION :: LOWER(*), DIAG(*), UPPER(*), RHS(*)
  INTEGER :: K, KB
  DOUBLE PRECISION :: T

  IF (N .EQ. 1) THEN
    IF (DIAG(1) .EQ. 0.0) GOTO 990
    RHS(1) = RHS(1)/DIAG(1)
  ENDIF
  LOWER(1) = DIAG(1)
  DIAG(1) = UPPER(1)
  UPPER(1) = 0.0
  UPPER(N) = 0.0
  DO K = 1, N-1
    ! Interchange this and next row to the get the largest pivot.
    IF (ABS(LOWER(K+1)) .GE. ABS(LOWER(K))) THEN
      T = LOWER(K+1)
      LOWER(K+1) = LOWER(K)
      LOWER(K) = T
      T = DIAG(K+1)
      DIAG(K+1) = DIAG(K)
      DIAG(K) = T
      T = UPPER(K+1)
      UPPER(K+1) = UPPER(K)
      UPPER(K) = T
      T = RHS(K+1)
      RHS(K+1) = RHS(K)
      RHS(K) = T
    ENDIF
    IF (LOWER(K) .EQ. 0.0) GOTO 990
    T = -LOWER(K+1)/LOWER(K)
    LOWER(K+1) = DIAG(K+1) + T*DIAG(K)
    DIAG(K+1) = UPPER(K+1) + T*UPPER(K)
    UPPER(K+1) = 0.0
    RHS(K+1) = RHS(K+1) + T*RHS(K)
  ENDDO
  IF (LOWER(N) .EQ. 0.0) GOTO 990
    ! Back substitute
  RHS(N) = RHS(N)/LOWER(N)
  RHS(N-1) = (RHS(N-1) - DIAG(N-1)*RHS(N))/LOWER(N-1)
  DO KB = 1, N-2
    K = N - 2 - KB + 1
    RHS(K) = (RHS(K) -DIAG(K)*RHS(K+1) -UPPER(K)*RHS(K+2))/LOWER(K)
  ENDDO
  RETURN
990 CONTINUE
    STOP 'Singular matrix in TRIDIAG'
END SUBROUTINE TRIDIAG
 
 



SUBROUTINE LEGENDRE_ALL (COSSCAT, NLEG, P)
 ! This subroutine computes a set of Legendre polynomials for
 ! a particular scattering angle COSSCAT.  NLEG is the maximum term.
 ! The Legendre functions evaluated at COSSCAT are returned in 
 ! P, starting at l=0 and ending with l=NLEG  (NLEG+1 terms).
  IMPLICIT NONE
      INTEGER NLEG
      DOUBLE PRECISION COSSCAT, P(0:NLEG)
      INTEGER L
      DOUBLE PRECISION X, PL, PL1, PL2

      X = COSSCAT
      IF (X*X .GT. 1.) STOP 'LEGENDRE_ALL: |COSSCAT| larger than 1'
      ! Use the stable upward recursion on l, starting from P_0
      PL2 = 1.0D0
      P(0) = PL2
      IF (NLEG .GT. 1) THEN
        PL1 = X
        P(1) = X
      ENDIF
      DO L = 2, NLEG
        PL = ( (2*L-1)*X*PL1 - (L-1)*PL2 )/L
        P(L) = PL
        PL2 = PL1
        PL1 = PL
      ENDDO
END SUBROUTINE LEGENDRE_ALL




SUBROUTINE YLMALL (MU, PHI, ML, MM, NCS, P)
 ! This subroutine computes a set of normalized spherical harmonic 
 ! functions, P(J), for a particular direction mu,phi. 
 ! ML is the maximum meridional mode, MM is the maximum azimuthal mode,
 ! and NCS is the azimuthal mode flag (|NCS|=1 for cosine only, |NCS|=2 for 
 ! sines and cosines).  Returns normalized associated Legendre functions 
 ! only if NCS<0. The set is returned for triangular truncation: 
 ! J = NCS*(L*(L+1))/2 + M+1  for L<=MM
 ! J = (NCS*MM+1)*L-MM*(2+NCS*(MM-1))/2 + M+1  for L>MM
  IMPLICIT NONE
      INTEGER ML, MM, NCS
      REAL    MU, PHI, P(*)
      INTEGER J, L, M, C
      DOUBLE PRECISION X, Y, A, PMM, PL, PL1, PL2, PHI8

      C = ABS(NCS)
      IF (C .NE. 1 .AND. C .NE. 2)  STOP 'YLMALL: bad NCS'
      IF (MM .GT. ML)  STOP 'YLMALL: MM greater than LM'
      IF (MU*MU .GT. 1.) STOP 'YLMALL: |MU| larger than 1'
      X = DBLE(MU)
      Y = SQRT(1.0D0-X*X)
      ! Use the stable upward recursion on l, starting from P^m_m
      ! Put in the spherical harmonic normalization as it goes
      PMM = 1.0D0/SQRT(4.0D0*ACOS(-1.0D0))
      DO M = 0, MM
        IF (M .GT. 0)  PMM = -PMM*Y*SQRT((2*M+1.0D0)/(2.0D0*M))
        J = C*(M*(M+1))/2 + M+1
        P(J) = PMM
        PL2 = PMM
        IF (M .LT. ML) THEN
          IF (M+1.LE.MM) J=C*((M+1)*(M+2))/2 +M+1
          IF (M+1.GT.MM) J=(C*MM+1)*(M+1)-(MM*(2+C*(MM-1)))/2+M+1
          PL1 = SQRT(2*M+3.0D0)*X*PMM
          P(J) = PL1
        ENDIF
        DO L = M+1, ML-1
          IF (L+1.LE.MM) J=C*((L+1)*(L+2))/2 +M+1
          IF (L+1.GT.MM) J=(C*MM+1)*(L+1)-(MM*(2+C*(MM-1)))/2+M+1
          A = 1.0D0/((L+M+1.D0)*(L-M+1.D0))
          PL = SQRT((2*L+1)*A*(2*L+3)) *X*PL1 &
                  - SQRT((2*L+3)*A*(L+M)*(L-M)/(2*L-1.)) *PL2
          P(J) = PL
          PL2 = PL1
          PL1 = PL
        ENDDO
        IF (M .EQ. 0) PMM = PMM*SQRT(2.0D0)
      ENDDO
      ! If there are M<0 terms then fill them in
      IF (C .EQ. 2) THEN
        DO L = 0, ML
          DO M = 1, MIN(L,MM)
            IF (L .LE. MM) J = L*(L+1) +M+1
            IF (L .GT. MM) J = MM*(2*L-MM) +L+M+1
            P(J-2*M) = P(J)
          ENDDO
        ENDDO
      ENDIF
      ! Put in the azimuthal dependence
      PHI8 = DBLE(PHI)
      IF (NCS .GT. 0) THEN
        DO M = (1-NCS)*MM, MM
          IF (M .LT. 0) THEN
            A = SIN(-M*PHI8)
          ELSE IF (M .GT. 0) THEN
            A = COS(M*PHI8)
          ELSE
            A = 1.0D0
          ENDIF
          DO L = ABS(M), ML
            IF (L .LE. MM) THEN
              J = C*(L*(L+1))/2 +M+1
            ELSE
              J = (C*MM+1)*L-(MM*(2+C*(MM-1)))/2 + M+1
            ENDIF
            P(J) = A*P(J)
          ENDDO
        ENDDO
      ENDIF
END SUBROUTINE YLMALL


 




SUBROUTINE GAUSQUADS (N, XA, WT)
 ! Generates the abscissas (X) and weights (W) for an N point
 ! Gauss-Legendre quadrature.  The XA are returned in this order: 
 ! -mu1, -mu2, ..., -muK, mu1, mu2, ..., muK  (mu1 > mu2 > muK, 
 ! K=N/2, N must be even).
  IMPLICIT NONE
      INTEGER  N
      REAL     XA(*), WT(*)
      INTEGER  K, I, J, L
      DOUBLE PRECISION  X, XP, PL, PL1, PL2, DPL, TINY
      PARAMETER (TINY=3.0D-7)

      K = N/2
      IF (2*K .NE. N) STOP 'GAUSQUADS: N must be even'
      DO J = 1, K
        X = COS(3.141592654*(J-.25)/(N+.5))
        I = 0
100     CONTINUE
          PL1 = 1
          PL = X
          DO L = 2, N
            PL2 = PL1
            PL1 = PL
            PL = ( (2*L-1)*X*PL1 - (L-1)*PL2 )/L
          ENDDO
          DPL = N*(X*PL-PL1)/(X*X-1)
          XP = X
          X = XP - PL/DPL
          I = I+1
        IF (ABS(X-XP).GT.TINY .AND. I.LT.10) GO TO 100
        XA(J)     = -X
        XA(J+K) = X
        WT(J  )   = 2.0D0/((1.0D0-X*X)*DPL*DPL)
        WT(J+K) = WT(J)
      ENDDO
END SUBROUTINE GAUSQUADS




SUBROUTINE DGAUSQUADS (N, XA, WT)
 ! Generates the abscissas (X) and weights (W) for an N point
 ! Double-Gauss-Legendre quadrature.  The XA are returned in this order: 
 ! -mu1, -mu2, ..., -muK, mu1, mu2, ..., muK  (mu1 > mu2 > muK, 
 ! K=N/2, N must be even).
  IMPLICIT NONE
      INTEGER  N
      REAL     XA(*), WT(*)
      INTEGER  K, I, J, L, N2
      DOUBLE PRECISION  X, XP, PL, PL1, PL2, DPL, TINY
      PARAMETER (TINY=3.0D-7)

      N2 = N/2
      IF (2*N2 .NE. N) STOP 'DGAUSQUADS: N must be even'
      K = (N2+1)/2
      DO J = 1, K
        X = COS(3.141592654*(J-.25)/(N2+.5))
        I = 0
100     CONTINUE
          PL1 = 1
          PL = X
          DO L = 2, N2
            PL2 = PL1
            PL1 = PL
            PL = ( (2*L-1)*X*PL1 - (L-1)*PL2 )/L
          ENDDO
          DPL = N2*(X*PL-PL1)/(X*X-1)
          XP = X
          X = XP - PL/DPL
          I = I+1
        IF (ABS(X-XP).GT.TINY .AND. I.LT.10) GO TO 100
        XA(J+N2) = (1+X)/2
        XA(N2+1-J+N2) = (1-X)/2
        WT(J+N2) = 1/((1-X*X)*DPL*DPL)
        WT(N2+1-J+N2) = WT(J+N2)
        XA(J) = -XA(J+N2)
        XA(N2+1-J) = -XA(N2+1-J+N2)
        WT(J) = WT(J+N2)
        WT(N2+1-J) = WT(J+N2)
      ENDDO
END SUBROUTINE DGAUSQUADS




SUBROUTINE INTEGRATE_PLANCK (X1, X2, F)
 ! Returns integral of x**3/(exp(x)-1) from x1 to x2.  Accurate to better 
 ! than 1 part in 10**9 for integral from 0 to infinity (pi**4/15).
  IMPLICIT NONE
  DOUBLE PRECISION :: X1, X2, F
  DOUBLE PRECISION :: INTLOW, INTHIGH, C

  C = 1.0D0
  IF (X1 < C .AND. X2 < C) THEN
    F = INTLOW(X2) - INTLOW(X1)
  ELSE IF (X1 < C .AND. X2 >= C) THEN
    F = INTLOW(C) - INTLOW(X1) + INTHIGH(C) - INTHIGH(X2)
  ELSE IF (X1 >= C .AND. X2 >= C) THEN
    F = INTHIGH(X1) - INTHIGH(X2)
  ELSE
    STOP 'X1 and X2 out of order'
  ENDIF
END SUBROUTINE INTEGRATE_PLANCK


FUNCTION INTLOW (X)
 ! Integral of x**3/(exp(x)-1) from 0 to x.  Accurate for x less than about 1.
 ! Uses Taylor series expansion around x=0.
  IMPLICIT NONE 
  DOUBLE PRECISION :: X, INTLOW
  INTEGER :: N
  DOUBLE PRECISION :: SUM, F(29), A(29)
  DATA    A/0.0D0, 0.0D0,  0.333333333333333333333D0, &
    -0.125D0,  0.016666666666666666667D0, 0.0D0, -0.0001984126984126984127D0, &
     0.0D0,  0.36743092298647854203D-5,   0.0D0, -0.75156325156325156325D-7, &
     0.0D0,  0.16059043836821614599D-8,   0.0D0, -0.35227934257916621232D-10, &
     0.0D0,  0.78720803121674581370D-12,  0.0D0, -0.17840422612224120352D-13, &
     0.0D0,  0.40886009791799259829D-15,  0.0D0, -0.94559508632959211871D-17, &
     0.0D0,  0.22036011313440918061D-18,  0.0D0, -0.51683202540046382743D-20, &
     0.0D0,  0.12188644964239543006D-20/

  SUM = A(4)*X**4
  F(1) = X
  DO N = 3, 29, 2
    F(N) = X*X*F(N-2)
    SUM = SUM + A(N)*F(N)
  ENDDO
  INTLOW = SUM
END FUNCTION INTLOW

FUNCTION INTHIGH (X)
 ! Integral of x**3/(exp(x)-1) from x to infinity.
 ! Accurate for x greater than about 1.
  IMPLICIT NONE
  DOUBLE PRECISION :: X, INTHIGH
  INTEGER :: N
  DOUBLE PRECISION :: SUM

  SUM = 0.0D0
  DO N = 1, 15
    SUM = SUM + EXP(-N*X) *(X**3/N + 3*X**2/N**2 + 6*X/N**3 + 6.0D0/N**4)
  ENDDO
  INTHIGH = SUM
END FUNCTION INTHIGH




      SUBROUTINE SSORT (X, Y, N, KFLAG)
! ***BEGIN PROLOGUE  SSORT
! ***PURPOSE  Sort an array and optionally make the same interchanges in
!             an auxiliary array.  The array may be sorted in increasing
!             or decreasing order.  A slightly modified QUICKSORT
!             algorithm is used.
! ***LIBRARY   SLATEC
! ***CATEGORY  N6A2B
! ***TYPE      SINGLE PRECISION (SSORT-S, DSORT-D, ISORT-I)
! ***KEYWORDS  SINGLETON QUICKSORT, SORT, SORTING
! ***AUTHOR  Jones, R. E., (SNLA)
!            Wisniewski, J. A., (SNLA)
! ***DESCRIPTION
! 
!    SSORT sorts array X and optionally makes the same interchanges in
!    array Y.  The array X may be sorted in increasing order or
!    decreasing order.  A slightly modified quicksort algorithm is used.
! 
!    Description of Parameters
!       X - array of values to be sorted   (usually abscissas)
!       Y - array to be (optionally) carried along
!       N - number of values in array X to be sorted
!       KFLAG - control parameter
!             =  2  means sort X in increasing order and carry Y along.
!             =  1  means sort X in increasing order (ignoring Y)
!             = -1  means sort X in decreasing order (ignoring Y)
!             = -2  means sort X in decreasing order and carry Y along.
! 
! ***REFERENCES  R. C. Singleton, Algorithm 347, An efficient algorithm
!                  for sorting with minimal storage, Communications of
!                  the ACM, 12, 3 (1969), pp. 185-187.
! ***END PROLOGUE  SSORT
!      .. Scalar Arguments ..
      INTEGER KFLAG, N
!      .. Array Arguments ..
!       REAL X(*), Y(*)
      REAL X(*)
      INTEGER Y(*)
!      .. Local Scalars ..
!       REAL R, T, TT, TTY, TY
      REAL R, T, TT
      INTEGER TY, TTY
      INTEGER I, IJ, J, K, KK, L, M, NN
!      .. Local Arrays ..
      INTEGER IL(51), IU(51)
!      .. External Subroutines ..
!      .. Intrinsic Functions ..
      INTRINSIC ABS, INT
! ***First executable statement  SSORT
      NN = N
      IF (NN .LT. 1) THEN
         STOP 'The number of values to be sorted is not positive.'
      ENDIF

      KK = ABS(KFLAG)
      IF (KK.NE.1 .AND. KK.NE.2) THEN
        STOP 'The sort control parameter, K, is not 2, 1, -1, or -2.'
      ENDIF

!      Alter array X to get decreasing order if needed
      IF (KFLAG .LE. -1) THEN
         DO 10 I=1,NN
            X(I) = -X(I)
   10    CONTINUE
      ENDIF

      IF (KK .EQ. 2) GO TO 100

!      Sort X only
      M = 1
      I = 1
      J = NN
      R = 0.375E0

   20 IF (I .EQ. J) GO TO 60
      IF (R .LE. 0.5898437E0) THEN
         R = R+3.90625E-2
      ELSE
         R = R-0.21875E0
      ENDIF

   30 K = I

!      Select a central element of the array and save it in location T
      IJ = I + INT((J-I)*R)
      T = X(IJ)

!      If first element of array is greater than T, interchange with T
      IF (X(I) .GT. T) THEN
         X(IJ) = X(I)
         X(I) = T
         T = X(IJ)
      ENDIF
      L = J

!      If last element of array is less than than T, interchange with T
      IF (X(J) .LT. T) THEN
         X(IJ) = X(J)
         X(J) = T
         T = X(IJ)

!         If first element of array is greater than T, interchange with T
         IF (X(I) .GT. T) THEN
            X(IJ) = X(I)
            X(I) = T
            T = X(IJ)
         ENDIF
      ENDIF

!      Find an element in the second half of the array which is smaller
!      than T
   40 L = L-1
      IF (X(L) .GT. T) GO TO 40

!      Find an element in the first half of the array which is greater
!      than T
   50 K = K+1
      IF (X(K) .LT. T) GO TO 50

!      Interchange these elements
      IF (K .LE. L) THEN
         TT = X(L)
         X(L) = X(K)
         X(K) = TT
         GO TO 40
      ENDIF

!      Save upper and lower subscripts of the array yet to be sorted
      IF (L-I .GT. J-K) THEN
         IL(M) = I
         IU(M) = L
         I = K
         M = M+1
      ELSE
         IL(M) = K
         IU(M) = J
         J = L
         M = M+1
      ENDIF
      GO TO 70

!      Begin again on another portion of the unsorted array
   60 M = M-1
      IF (M .EQ. 0) GO TO 190
      I = IL(M)
      J = IU(M)

   70 IF (J-I .GE. 1) GO TO 30
      IF (I .EQ. 1) GO TO 20
      I = I-1

   80 I = I+1
      IF (I .EQ. J) GO TO 60
      T = X(I+1)
      IF (X(I) .LE. T) GO TO 80
      K = I

   90 X(K+1) = X(K)
      K = K-1
      IF (T .LT. X(K)) GO TO 90
      X(K+1) = T
      GO TO 80

!      Sort X and carry Y along
  100 M = 1
      I = 1
      J = NN
      R = 0.375E0

  110 IF (I .EQ. J) GO TO 150
      IF (R .LE. 0.5898437E0) THEN
         R = R+3.90625E-2
      ELSE
         R = R-0.21875E0
      ENDIF
! 
  120 K = I

!      Select a central element of the array and save it in location T
      IJ = I + INT((J-I)*R)
      T = X(IJ)
      TY = Y(IJ)

!      If first element of array is greater than T, interchange with T
      IF (X(I) .GT. T) THEN
         X(IJ) = X(I)
         X(I) = T
         T = X(IJ)
         Y(IJ) = Y(I)
         Y(I) = TY
         TY = Y(IJ)
      ENDIF
      L = J

!      If last element of array is less than T, interchange with T
      IF (X(J) .LT. T) THEN
         X(IJ) = X(J)
         X(J) = T
         T = X(IJ)
         Y(IJ) = Y(J)
         Y(J) = TY
         TY = Y(IJ)

!         If first element of array is greater than T, interchange with T
         IF (X(I) .GT. T) THEN
            X(IJ) = X(I)
            X(I) = T
            T = X(IJ)
            Y(IJ) = Y(I)
            Y(I) = TY
            TY = Y(IJ)
         ENDIF
      ENDIF

!      Find an element in the second half of the array which is smaller
!      than T
  130 L = L-1
      IF (X(L) .GT. T) GO TO 130

!      Find an element in the first half of the array which is greater
!      than T
  140 K = K+1
      IF (X(K) .LT. T) GO TO 140

!      Interchange these elements
      IF (K .LE. L) THEN
         TT = X(L)
         X(L) = X(K)
         X(K) = TT
         TTY = Y(L)
         Y(L) = Y(K)
         Y(K) = TTY
         GO TO 130
      ENDIF

!      Save upper and lower subscripts of the array yet to be sorted
      IF (L-I .GT. J-K) THEN
         IL(M) = I
         IU(M) = L
         I = K
         M = M+1
      ELSE
         IL(M) = K
         IU(M) = J
         J = L
         M = M+1
      ENDIF
      GO TO 160

!      Begin again on another portion of the unsorted array
  150 M = M-1
      IF (M .EQ. 0) GO TO 190
      I = IL(M)
      J = IU(M)

  160 IF (J-I .GE. 1) GO TO 120
      IF (I .EQ. 1) GO TO 110
      I = I-1

  170 I = I+1
      IF (I .EQ. J) GO TO 150
      T = X(I+1)
      TY = Y(I+1)
      IF (X(I) .LE. T) GO TO 170
      K = I

  180 X(K+1) = X(K)
      Y(K+1) = Y(K)
      K = K-1
      IF (T .LT. X(K)) GO TO 180
      X(K+1) = T
      Y(K+1) = TY
      GO TO 170

!      Clean up
  190 IF (KFLAG .LE. -1) THEN
         DO 200 I=1,NN
            X(I) = -X(I)
  200    CONTINUE
      ENDIF
      RETURN
      END




SUBROUTINE READ_SCAT_TABLE_SIZE (SCATTABFILE, WAVELEN1, WAVELEN2, &
                                 PARDENS, NRTAB, MAXNLEG)
 ! Reads the scattering table file to find the number of radii,
 ! the maximum order of the Legendre series phase functions, and also
 ! the wavelength range.
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN) :: SCATTABFILE
  INTEGER, INTENT(OUT) :: NRTAB, MAXNLEG
  REAL,    INTENT(OUT) :: WAVELEN1, WAVELEN2, PARDENS
  INTEGER :: I, J, L, NLEG
  REAL    :: RMM, RE, EXT, SSALB, CHI

  OPEN (UNIT=1, FILE=SCATTABFILE, STATUS='OLD')
  READ (1,*)
  READ (1,*) WAVELEN1, WAVELEN2
  READ (1,*) PARDENS
  READ (1,*)
  READ (1,*)
  READ (1,*) NRTAB
  MAXNLEG = 1
  DO I = 1, NRTAB
    READ (1,*) RMM, RE, EXT, SSALB, NLEG
    MAXNLEG = MAX(MAXNLEG,NLEG)
    READ (1,*) (CHI, L=0,NLEG)
  ENDDO
  CLOSE (1)
END SUBROUTINE READ_SCAT_TABLE_SIZE



SUBROUTINE READ_SCAT_TABLE (SCATTABFILE, NRTAB, RTAB, EXTINCT, SSALB, &
                            MAXLEG, NLEG, LEGCOEF)
 ! Reads the table of scattering properties as a function of radius (RTAB).
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN) :: SCATTABFILE
  INTEGER, INTENT(IN)  :: NRTAB, MAXLEG
  INTEGER, INTENT(OUT) :: NLEG(NRTAB)
  REAL,    INTENT(OUT) :: RTAB(NRTAB), EXTINCT(NRTAB), SSALB(NRTAB)
  REAL,    INTENT(OUT) :: LEGCOEF(1:MAXLEG,NRTAB)
  INTEGER :: I, J, L
  REAL    :: RE, CHI0

  OPEN (UNIT=1, FILE=SCATTABFILE, STATUS='OLD')
  READ (1,*)
  READ (1,*)
  READ (1,*)
  READ (1,*)
  READ (1,*)
  READ (1,*) 
  DO I = 1, NRTAB
    READ (1,*) RTAB(I), RE, EXTINCT(I), SSALB(I), NLEG(I)
    LEGCOEF(1:MAXLEG,I) = 0.0
    READ (1,*) CHI0, (LEGCOEF(L,I), L=1,NLEG(I))
    IF (ABS(CHI0-1.0) > 0.0001) THEN
      PRINT *, 'READ_SCAT_TABLE: Incorrect Legendre series; chi_0 is not 1'
      STOP
    ENDIF
    IF (I > 1 .AND. RTAB(I) <= RTAB(I-1)) THEN
      PRINT *,'READ_SCAT_TABLE: Radius not increasing in table:',SCATTABFILE
      STOP
    ENDIF
  ENDDO
  CLOSE (1)
END SUBROUTINE READ_SCAT_TABLE


